import numpy as np
from env import setup_environment, shutdown_environment
from skill_code import pick, place, move, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # reset and start recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # get all named positions in the scene
        positions = get_object_positions()
        print("[Positions] ", positions)

        # identify the relevant positions/objects for opening the drawer
        drawer_key = next((k for k in positions if "drawer" in k.lower()), None)
        side_key   = next((k for k in positions if "side"   in k.lower()), None)
        anchor_key = next((k for k in positions if "anchor" in k.lower()), None)
        handle_key = next((k for k in positions if "handle" in k.lower()), None)

        if None in (drawer_key, side_key, anchor_key, handle_key):
            raise RuntimeError("Could not find all required keys: "
                               f"drawer={drawer_key}, side={side_key}, "
                               f"anchor={anchor_key}, handle={handle_key}")

        side_pos   = positions[side_key]
        anchor_pos = positions[anchor_key]
        handle_pos = positions[handle_key]

        # 1) Rotate the gripper so it can approach the side of the drawer
        print("[Task] Rotating gripper to ninety_deg")
        obs, reward, done = rotate(env, task, "ninety_deg")
        if done:
            print("[Task] Early termination after rotate")
            return

        # 2) Move to the side position of the drawer
        print(f"[Task] Moving gripper to side position of drawer at {side_pos}")
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Task] Early termination after moving to side")
            return

        # 3) Move from the side position to the anchor position on the drawer
        print(f"[Task] Moving gripper to anchor position of drawer at {anchor_pos}")
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Task] Early termination after moving to anchor")
            return

        # 4) Pick the drawer handle
        print(f"[Task] Picking the drawer handle at {handle_pos}")
        obs, reward, done = pick(env, task, handle_pos)
        if done:
            print("[Task] Early termination after pick")
            return

        # 5) Pull to open the drawer
        print("[Task] Pulling to open the drawer")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Early termination after pull")
            return

        print("[Task] Drawer opening sequence completed successfully.")

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()