# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Exploration and Plan Execution ===
        # We attempt to open a drawer and discover the missing predicate (lock-known).
        try:
            # 1) Rotate gripper to ninety degrees for side alignment
            print("[Task] Rotating gripper to 90 degrees for side alignment")
            ninety_deg_quat = [0.0, 0.0, 0.7071, 0.7071]
            obs, reward, done = rotate(env, task, ninety_deg_quat)
            if done:
                print("[Task] Early termination after rotation.")
                return

            # 2) Move gripper to the side position of the drawer
            side_pos = positions.get('drawer_side_pos')
            if side_pos is None:
                raise KeyError("drawer_side_pos not found in positions")
            print(f"[Task] Moving to drawer side position: {side_pos}")
            obs, reward, done = move(env, task, side_pos)
            if done:
                print("[Task] Early termination after moving to side.")
                return

            # 3) Move gripper to the anchor position to grasp the drawer handle
            anchor_pos = positions.get('drawer_anchor_pos')
            if anchor_pos is None:
                raise KeyError("drawer_anchor_pos not found in positions")
            print(f"[Task] Moving to drawer anchor position: {anchor_pos}")
            obs, reward, done = move(env, task, anchor_pos)
            if done:
                print("[Task] Early termination after moving to anchor.")
                return

            # 4) Pick the drawer handle
            print("[Task] Picking drawer handle")
            obs, reward, done = pick(env, task, anchor_pos)
            if done:
                print("[Task] Early termination after pick.")
                return

            # 5) Pull the drawer to identify the lock state (lock-known predicate)
            print("[Task] Pulling the drawer to identify lock state")
            obs, reward, done = pull(env, task)
            if done:
                print("[Task] Early termination after pull.")
                return

            print("[Task] Completed drawer pull. Missing predicate 'lock-known' has been identified.")
        except Exception as e:
            print(f"[Task] Exception during execution: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()