import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import rotate, move, pick, place, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # retrieve positions of relevant features
        positions = get_object_positions()
        # these keys must match what object_positions provides
        # side position adjacent to drawer
        side_pos = positions['drawer_side_pos']
        # anchor position at drawer handle
        anchor_pos = positions['drawer_anchor_pos']
        # quaternion for a 90° rotation around the x‑axis (xyzw)
        ninety_deg_quat = np.array([0.7071, 0.0, 0.0, 0.7071])

        # 1) Rotate the gripper so that it is aligned for grasping the drawer handle
        print("[Task] Rotating gripper to ninety degrees")
        obs, reward, done = rotate(env, task, ninety_deg_quat)
        if done:
            print("[Task] Finished early during rotation.")
            return

        # 2) Move to the side of the drawer
        print("[Task] Moving to side of drawer")
        obs, reward, done = move(
            env, task,
            target_pos=side_pos,
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Finished early during side approach.")
            return

        # 3) Move into the anchor position (the drawer handle)
        print("[Task] Moving to drawer handle (anchor position)")
        obs, reward, done = move(
            env, task,
            target_pos=anchor_pos,
            approach_distance=0.05,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Finished early during anchor approach.")
            return

        # 4) Pick the drawer handle
        print("[Task] Picking the drawer handle")
        obs, reward, done = pick(
            env, task,
            target_pos=anchor_pos,
            approach_distance=0.02,
            max_steps=100,
            threshold=0.005,
            approach_axis='x',
            timeout=5.0
        )
        if done:
            print("[Task] Finished early during pick.")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling to open the drawer")
        obs, reward, done = pull(
            env, task,
            target_pos=anchor_pos,
            approach_distance=0.00,
            max_steps=100,
            threshold=0.005,
            approach_axis='x',
            timeout=5.0
        )
        if done:
            print("[Task] Finished early during pull.")
            return

        print("===== Task Completed: Drawer should now be open =====")

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()