import numpy as np
from env import setup_environment, shutdown_environment
from skill_code import rotate, move_to_side, move_to_anchor, pick_drawer, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # get object positions
        positions = get_object_positions()
        # identify the gripper and drawer names
        if 'gripper1' not in positions:
            raise KeyError("Expected key 'gripper1' in object positions")
        gripper_name = 'gripper1'
        gripper_pos = positions[gripper_name]
        # find side and anchor positions for the drawer
        side_key = None
        anchor_key = None
        for k in positions:
            if k.endswith('_side'):
                side_key = k
                base = k[:-5]
                anchor_key = base + '_anchor'
                break
        if side_key is None or anchor_key not in positions:
            raise KeyError("Could not find side/anchor positions for drawer in positions")
        side_pos = positions[side_key]
        anchor_pos = positions[anchor_key]
        # infer drawer_name from key naming convention
        drawer_name = side_key[:-5]

        # 1) rotate gripper from zero_deg to ninety_deg
        zero_quat = [1.0, 0.0, 0.0, 0.0]   # identity quaternion
        ninety_quat = [0.0, 0.7071, 0.0, 0.7071]  # 90° about Y axis as example
        print(f"[Task] Rotating gripper to zero orientation")
        obs, reward, done = rotate(env, task, zero_quat)
        if done:
            print("[Task] Task ended after zero rotation")
            return
        print(f"[Task] Rotating gripper to ninety-degree orientation")
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("[Task] Task ended after ninety rotation")
            return

        # 2) move gripper to side position of drawer
        print(f"[Task] Moving to side position of {drawer_name}: {side_pos}")
        obs, reward, done = move_to_side(env, task, gripper_name, drawer_name, gripper_pos, side_pos)
        if done:
            print("[Task] Task ended after move_to_side")
            return

        # 3) move gripper from side to anchor position
        print(f"[Task] Moving to anchor position of {drawer_name}: {anchor_pos}")
        obs, reward, done = move_to_anchor(env, task, gripper_name, drawer_name, side_pos, anchor_pos)
        if done:
            print("[Task] Task ended after move_to_anchor")
            return

        # 4) pick the drawer handle
        print(f"[Task] Picking drawer handle on {drawer_name}")
        obs, reward, done = pick_drawer(env, task, gripper_name, drawer_name, anchor_pos)
        if done:
            print("[Task] Task ended after pick_drawer")
            return

        # 5) pull the drawer open
        print(f"[Task] Pulling open {drawer_name}")
        obs, reward, done = pull(env, task, gripper_name, drawer_name)
        if done:
            print("[Task] Task ended after pull")
            return

        print("===== Drawer open sequence complete =====")

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()