import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # get the 3D positions of all objects in the scene
        positions = get_object_positions()
        print("[Info] Retrieved object positions:", positions.keys())

        # Exploration phase: move to every known object to identify missing predicates
        for name, pos in positions.items():
            print(f"[Exploration] Moving to position of {name}: {pos}")
            obs, reward, done = move(env, task, pos)
            if done:
                print("[Exploration] Task ended prematurely during exploration.")
                return

        # Identify side and anchor positions for the drawer from the key names
        side_key = next(k for k in positions if 'side' in k and 'drawer' in k)
        anchor_key = next(k for k in positions if 'anchor' in k and 'drawer' in k)
        side_pos = positions[side_key]
        anchor_pos = positions[anchor_key]
        print(f"[Plan] side_pos='{side_key}' at {side_pos}, anchor_pos='{anchor_key}' at {anchor_pos}")

        # 1) Rotate the gripper to a 90‐degree orientation so we can move to the side position
        #    Here we choose a quaternion representing a 90° rotation around Z.
        ninety_deg_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])
        obs, reward, done = rotate(env, task, ninety_deg_quat)
        if done:
            print("[Plan] Task ended during rotate(90°).")
            return

        # 2) Move the gripper to the side position of the drawer
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Plan] Task ended during move to side_pos.")
            return

        # 3) Move from side position to anchor position (handle) on the drawer
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Plan] Task ended during move to anchor_pos.")
            return

        # 4) Pick the drawer handle (anchor position)
        obs, reward, done = pick(
            env,
            task,
            target_pos=anchor_pos,
            approach_distance=0.02,
            max_steps=100,
            threshold=0.005,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Plan] Task ended during pick(handle).")
            return

        # 5) Pull the drawer open
        obs, reward, done = pull(
            env,
            task,
            # assume pull skill requires only env and task to act on the held handle
            max_steps=50,
            threshold=0.01,
            timeout=5.0
        )
        if done:
            print("[Plan] Task ended during pull.")
            return

        # 6) (Optional) rotate back to home orientation
        home_quat = np.array([0.0, 0.0, 0.0, 1.0])
        obs, reward, done = rotate(env, task, home_quat)
        if done:
            print("[Plan] Task ended during rotate(home).")
            return

        print("===== Task plan completed successfully! =====")

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()