import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Replace these keys with the actual names in your environment
        gripper_name = 'gripper'
        drawer_name = 'drawer'
        side_pos = positions.get('drawer_side_pos')
        anchor_pos = positions.get('drawer_anchor_pos')

        if side_pos is None or anchor_pos is None:
            print("ERROR: Missing positional information for the drawer.")
            return

        # === Oracle Plan Execution ===

        # 1) Rotate the gripper to 90 degrees so it can approach the drawer handle
        print("[Task] Rotating gripper to 90° orientation")
        target_quat = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Terminated during rotate")
            return

        # 2) Move the gripper to the "side" position of the drawer
        print("[Task] Moving to drawer side position")
        obs, reward, done = move_to_side(env, task, gripper_name, drawer_name, side_pos)
        if done:
            print("[Task] Terminated during move_to_side")
            return

        # 3) Move the gripper from the side position to the anchor position (handle)
        print("[Task] Moving to drawer anchor (handle) position")
        obs, reward, done = move_to_anchor(env, task, gripper_name, drawer_name, anchor_pos)
        if done:
            print("[Task] Terminated during move_to_anchor")
            return

        # 4) Grasp the drawer handle
        print("[Task] Grasping the drawer handle")
        obs, reward, done = pick_drawer(env, task, gripper_name, drawer_name, anchor_pos)
        if done:
            print("[Task] Terminated during pick_drawer")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling the drawer open")
        obs, reward, done = pull(env, task, gripper_name, drawer_name)
        if done:
            print("[Task] Terminated during pull")
            return

        print("[Task] Drawer successfully opened!")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()