import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers (optional)
        init_video_writers(obs)

        # Wrap step/get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve positions of objects in the scene
        positions = get_object_positions()

        # Define the object we want to manipulate and where to place it
        target_object = 'rubbish'
        placement_target = 'bin'

        # Safety check: ensure the object is in the environment
        object_list = task.get_objects()
        if target_object not in object_list:
            print(f"Error: Object '{target_object}' not found in object list: {object_list}")
            raise ValueError(f"Object '{target_object}' not found")

        # Safety check: ensure we have pose information
        if target_object not in positions:
            print(f"Error: Position for '{target_object}' not found in positions: {positions.keys()}")
            raise ValueError(f"Position for '{target_object}' missing")
        if placement_target not in positions:
            print(f"Error: Position for placement target '{placement_target}' not found in positions: {positions.keys()}")
            raise ValueError(f"Position for '{placement_target}' missing")

        object_pos = positions[target_object]
        bin_pos = positions[placement_target]

        # === Oracle Plan Execution ===
        # 1) Pick up the rubbish
        print(f"[Task] Picking up '{target_object}' at {object_pos}")
        obs, reward, done = pick(
            env,
            task,
            target_pos=object_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended prematurely after pick.")
            return

        # 2) Place the rubbish into the bin
        print(f"[Task] Placing '{target_object}' into '{placement_target}' at {bin_pos}")
        obs, reward, done = place(
            env,
            task,
            target_pos=bin_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended prematurely after place.")
            return

        print("[Task] Completed pick-and-place successfully.")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()