import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # retrieve object positions
        positions = get_object_positions()
        # assume keys 'drawer_side_pos' and 'drawer_anchor_pos' exist
        if 'drawer_side_pos' not in positions or 'drawer_anchor_pos' not in positions:
            raise KeyError("Expected keys 'drawer_side_pos' and 'drawer_anchor_pos' in positions")

        drawer_side_pos = positions['drawer_side_pos']
        drawer_anchor_pos = positions['drawer_anchor_pos']

        # 1) Rotate gripper to ninety-degree orientation before approaching drawer
        # Here we compute a quaternion for 90 degrees around the vertical axis
        # (x, y, z, w) convention
        angle = np.pi / 2
        # quaternion for rotation about z: [0, 0, sin(angle/2), cos(angle/2)]
        target_quat = np.array([0.0, 0.0, np.sin(angle/2), np.cos(angle/2)])
        print("[Task] Rotating gripper to ninety_deg")
        obs, reward, done = rotate(env, task, target_quat, max_steps=100, threshold=0.02, timeout=8.0)
        if done:
            print("[Task] Task ended during rotate")
            return

        # 2) Move gripper to side position of drawer
        print(f"[Task] Moving gripper to drawer side pos: {drawer_side_pos}")
        obs, reward, done = move(env, task, drawer_side_pos)
        if done:
            print("[Task] Task ended during move to side")
            return

        # 3) Move gripper to anchor handle position
        print(f"[Task] Moving gripper to drawer anchor pos: {drawer_anchor_pos}")
        obs, reward, done = move(env, task, drawer_anchor_pos)
        if done:
            print("[Task] Task ended during move to anchor")
            return

        # 4) Pick the drawer handle
        print("[Task] Picking drawer handle")
        obs, reward, done = pick(env, task, drawer_anchor_pos, approach_distance=0.05, max_steps=50, threshold=0.01, approach_axis='z', timeout=5.0)
        if done:
            print("[Task] Task ended during pick")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling drawer open")
        obs, reward, done = pull(env, task, drawer_anchor_pos, pull_distance=0.2, max_steps=60, threshold=0.01, timeout=7.0)
        if done:
            print("[Task] Task ended during pull")
            return

        print("===== Task plan executed successfully =====")

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()