import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Expect keys like 'side_pos', 'anchor_pos', and possibly 'ninety_quat'
        side_pos = positions.get('side_pos')
        anchor_pos = positions.get('anchor_pos')
        ninety_quat = positions.get('ninety_quat')  # quaternion for 90° rotation
        if side_pos is None or anchor_pos is None or ninety_quat is None:
            raise KeyError("Required positions or quaternion missing from object_positions")

        # === Oracle Plan Execution ===

        # 1) Rotate the gripper to align with drawer handle
        print("[Task] Rotating gripper to 90 degrees")
        try:
            obs, reward, done = rotate(env, task, target_quat=ninety_quat)
        except Exception as e:
            print(f"[Error] rotate failed: {e}")
            return
        if done:
            print("[Task] Episode ended during rotate")
            return

        # 2) Move gripper to side-of-drawer position
        print("[Task] Moving to side position:", side_pos)
        try:
            obs, reward, done = move(env, task, from_pos=obs.gripper_pose[:3], to_pos=side_pos)
        except Exception as e:
            print(f"[Error] move to side failed: {e}")
            return
        if done:
            print("[Task] Episode ended during move to side")
            return

        # 3) Move gripper to anchor (handle) position
        print("[Task] Moving to anchor position:", anchor_pos)
        try:
            obs, reward, done = move(env, task, from_pos=obs.gripper_pose[:3], to_pos=anchor_pos)
        except Exception as e:
            print(f"[Error] move to anchor failed: {e}")
            return
        if done:
            print("[Task] Episode ended during move to anchor")
            return

        # 4) Pick the drawer handle
        print("[Task] Picking drawer handle")
        try:
            obs, reward, done = pick(env, task, obj='drawer_handle', loc=anchor_pos)
        except Exception as e:
            print(f"[Error] pick handle failed: {e}")
            return
        if done:
            print("[Task] Episode ended during pick handle")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling drawer")
        try:
            obs, reward, done = pull(env, task)
        except Exception as e:
            print(f"[Error] pull failed: {e}")
            return
        if done:
            print("[Task] Episode ended during pull")
            return

        print("[Task] Drawer should now be open")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()