import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # reset the scene
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # get all relevant positions in the scene
        positions = get_object_positions()
        # these keys must match what object_positions provides
        # side‐of‐drawer where gripper approaches first
        drawer_side_pos = positions.get('drawer_side_pos') or positions.get('side_position')
        # anchor position on drawer handle for picking
        drawer_anchor_pos = positions.get('drawer_anchor_pos') or positions.get('anchor_position')

        if drawer_side_pos is None or drawer_anchor_pos is None:
            raise KeyError("Missing side or anchor positions for the drawer in get_object_positions()")

        # 1) Move gripper to side position (pre‐grasp pose)
        print("[Task] Moving to drawer side position:", drawer_side_pos)
        obs, reward, done = move(env, task, drawer_side_pos)
        if done:
            print("[Task] Episode ended after move-to-side")
            return

        # 2) Rotate gripper from its current angle to ninety_deg
        # Here we assume starting from 'zero_deg'
        print("[Task] Rotating gripper to ninety_deg")
        obs, reward, done = rotate(env, task, 'zero_deg', 'ninety_deg')
        if done:
            print("[Task] Episode ended after rotate")
            return

        # 3) Move gripper to anchor position (aligned with drawer handle)
        print("[Task] Moving to drawer anchor position:", drawer_anchor_pos)
        obs, reward, done = move(env, task, drawer_anchor_pos)
        if done:
            print("[Task] Episode ended after move-to-anchor")
            return

        # 4) Pick the drawer handle
        print("[Task] Picking the drawer handle")
        obs, reward, done = pick(
            env,
            task,
            target_pos=drawer_anchor_pos,
            approach_distance=0.02,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Episode ended after pick")
            return

        # 5) Pull to open the drawer
        print("[Task] Pulling the drawer open")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Episode ended after pull")
            return

        print("[Task] Completed all plan steps successfully.")

    except Exception as e:
        print("[Task] Exception encountered:", e)
    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()