import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        if not positions:
            print("[Error] No object positions returned by get_object_positions()")
            return

        # === Exploration Phase ===
        # We will move to each object, pick it up, inspect it, then place it back.
        # This will trigger weight-known or durability-known predicates in the PDDL domain.
        for name, pos in positions.items():
            print(f"[Exploration] Inspecting '{name}' at position {pos}")

            # Step 1: Move the robot to the object location
            try:
                obs, reward, done = move(env, task, pos)
            except Exception as e:
                print(f"[Error] move failed for '{name}': {e}")
                continue
            if done:
                print("[Task] Task ended prematurely after move")
                return

            # Step 2: Pick the object (this also reveals weight or durability)
            # Ensure hand is empty before pick
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=5.0
                )
            except Exception as e:
                print(f"[Error] pick failed for '{name}': {e}")
                continue
            if done:
                print("[Task] Task ended prematurely after pick")
                return

            # Step 3: Place the object back at its original spot to free the gripper
            try:
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=5.0
                )
            except Exception as e:
                print(f"[Error] place failed for '{name}': {e}")
                continue
            if done:
                print("[Task] Task ended prematurely after place")
                return

        print("[Exploration] Completed inspection of all objects")

        # === Identify Missing Predicate ===
        # After exploring all objects through pick/place,
        # we have triggered the weight-known and durability-known effects.
        # Now we inspect one more object to discover lock-known via pull.
        # For simplicity, choose the first object in positions.
        first_name = next(iter(positions.keys()))
        handle_pos = positions[first_name]
        print(f"[Discovery] Attempting pull on '{first_name}' to uncover lock status")

        # Move to the handle position again
        obs, reward, done = move(env, task, handle_pos)
        if done:
            print("[Task] Task ended prematurely before pull")
            return

        # Pick the handle (assumes same pick skill works for the drawer handle)
        obs, reward, done = pick(
            env,
            task,
            target_pos=handle_pos,
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=5.0
        )
        if done:
            print("[Task] Task ended prematurely before pull")
            return

        # Finally, pull to learn the lock-known predicate
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Task ended prematurely after pull")
            return

        print("[Result] lock-known predicate has been discovered through pull")
        print("===== End of Skeleton Task =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

if __name__ == "__main__":
    run_skeleton_task()