import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        positions = get_object_positions()
        # these keys must match whatever object_positions provides
        side_pos = positions['drawer_side_pos']
        anchor_pos = positions['drawer_anchor_pos']

        # define a 90-degree gripper rotation quaternion (xyzw)
        ninety_deg_quat = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]

        # 1) Rotate gripper so it can later approach the drawer side
        print("[Task] Rotating gripper to 90 degrees around relevant axis")
        obs, reward, done = rotate(env, task, ninety_deg_quat)
        if done:
            print("[Task] Finished during rotate step.")
            return

        # 2) Move gripper to the 'side' position of the drawer
        print(f"[Task] Moving to drawer side position: {side_pos}")
        obs, reward, done = move(env, task, target_pos=side_pos)
        if done:
            print("[Task] Finished during move-to-side step.")
            return
        
        # 3) Move gripper to the 'anchor' position near the drawer handle
        print(f"[Task] Moving to drawer anchor position: {anchor_pos}")
        obs, reward, done = move(env, task, target_pos=anchor_pos)
        if done:
            print("[Task] Finished during move-to-anchor step.")
            return

        # 4) Pick (grasp) the drawer handle at the anchor position
        print("[Task] Grasping drawer handle")
        obs, reward, done = pick(env, task, target_pos=anchor_pos)
        if done:
            print("[Task] Finished during pick-handle step.")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling the drawer open")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Finished during pull step.")
            return

        print("[Task] Plan completed without early termination.")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()