import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # We assume the drawer handle and anchor positions are provided by object_positions
        # Example keys might be 'drawer_side_pos' and 'drawer_anchor_pos'
        try:
            side_pos = positions['drawer_side_pos']
            anchor_pos = positions['drawer_anchor_pos']
        except KeyError:
            raise RuntimeError("Missing required position keys in object_positions().")

        # Define identifiers for the gripper and drawer
        gripper_name = 'gripper'
        drawer_name = 'drawer'

        # === Oracle Plan Execution ===
        # 1) Rotate the gripper from its default angle to ninety_deg
        from_angle = 'zero_deg'
        to_angle = 'ninety_deg'
        print(f"[Task] Rotating {gripper_name} from {from_angle} to {to_angle}")
        obs, reward, done = rotate(env, task, gripper_name, from_angle, to_angle)
        if done:
            print("[Task] Terminated during rotate.")
            return

        # 2) Move gripper to the side position of the drawer
        print(f"[Task] Moving {gripper_name} to side position of {drawer_name} at {side_pos}")
        obs, reward, done = move_to_side(env, task, gripper_name, drawer_name, side_pos)
        if done:
            print("[Task] Terminated during move-to-side.")
            return

        # 3) Move gripper from side position to anchor position on the drawer
        print(f"[Task] Moving {gripper_name} from side to anchor position of {drawer_name} at {anchor_pos}")
        obs, reward, done = move_to_anchor(env, task, gripper_name, drawer_name, side_pos, anchor_pos)
        if done:
            print("[Task] Terminated during move-to-anchor.")
            return

        # 4) Pick the drawer handle
        print(f"[Task] Picking drawer handle with {gripper_name}")
        obs, reward, done = pick_drawer(env, task, gripper_name, drawer_name, anchor_pos)
        if done:
            print("[Task] Terminated during pick-drawer.")
            return

        # 5) Pull the handle to open the drawer
        pull_distance = 0.1  # adjust as necessary for your drawer
        print(f"[Task] Pulling {drawer_name} by {pull_distance}m along x-axis")
        obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
        if done:
            print("[Task] Terminated during pull.")
            return

        print("[Task] Drawer open procedure completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()