import numpy as np
from env import setup_environment, shutdown_environment
from skill_code import rotate, move, pick, pull, normalize_quaternion
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # reset to initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # retrieve object and position info
        positions = get_object_positions()
        # assume keys in positions dict for drawer side and anchor
        side_pos = positions.get('drawer_side_pos')
        anchor_pos = positions.get('drawer_anchor_pos')

        # prepare target orientations: keep current zero and target ninety degrees
        obs = task.get_observation()
        current_quat = normalize_quaternion(obs.gripper_pose[3:7])
        # build a 90° rotation around z axis
        ninety_quat = normalize_quaternion(np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]))
        # if gripper is not already at ninety_deg, rotate it
        dot = abs(np.dot(current_quat, ninety_quat))
        if 1.0 - dot > 1e-3:
            print("[Task] Rotating gripper from current to 90° orientation")
            obs, reward, done = rotate(env, task, ninety_quat)
            if done:
                print("[Task] Ended during rotate")
                return

        # move to drawer side position
        if side_pos is None:
            raise KeyError("Missing 'drawer_side_pos' in positions")
        print(f"[Task] Moving to side position: {side_pos}")
        obs, reward, done = move(env, task, target_pos=side_pos)
        if done:
            print("[Task] Ended during move to side")
            return

        # move to drawer anchor (handle) position
        if anchor_pos is None:
            raise KeyError("Missing 'drawer_anchor_pos' in positions")
        print(f"[Task] Moving to anchor position: {anchor_pos}")
        obs, reward, done = move(env, task, target_pos=anchor_pos)
        if done:
            print("[Task] Ended during move to anchor")
            return

        # pick the drawer handle
        print("[Task] Picking drawer handle")
        obs, reward, done = pick(env, task, target_pos=anchor_pos)
        if done:
            print("[Task] Ended during pick")
            return

        # pull to open the drawer
        print("[Task] Pulling drawer open")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Ended during pull")
            return

        print("[Task] Drawer opened successfully")

    except Exception as e:
        print(f"[Task] Exception encountered: {e}")
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()