import numpy as np
from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve object positions ===
        positions = get_object_positions()
        # we expect keys like 'drawer_handle', 'table_surface', maybe 'inside_object'
        drawer_handle_pos = positions.get('drawer_handle')
        table_surface_pos = positions.get('table_surface')
        inside_object_pos = positions.get('inside_object')

        if drawer_handle_pos is None:
            print("[Error] No drawer_handle in positions dictionary.")
            return

        # === 1) Rotate gripper so that it can approach the drawer handle ===
        # we assume initial rotation is zero_deg; we want ninety_deg around the vertical axis
        ninety_deg_quat = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]
        print("[Task] Rotating gripper to 90 degrees orientation")
        obs, reward, done = rotate(env, task, ninety_deg_quat)
        if done:
            print("[Task] Task ended during rotation.")
            return

        # === 2) Move gripper to the drawer handle position ===
        print(f"[Task] Moving gripper to drawer handle at {drawer_handle_pos}")
        obs, reward, done = move(env, task, target_pos=drawer_handle_pos)
        if done:
            print("[Task] Task ended during move-to-handle.")
            return

        # === 3) Pick the drawer handle ===
        print("[Task] Picking the drawer handle")
        obs, reward, done = pick(env, task, target_pos=drawer_handle_pos)
        if done:
            print("[Task] Task ended during pick-handle.")
            return

        # === 4) Pull the drawer open ===
        print("[Task] Pulling the drawer open")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Task ended during pull.")
            return

        # === 5) If there is an object inside, remove it and place on table ===
        if inside_object_pos is not None and table_surface_pos is not None:
            print(f"[Task] Removing object from drawer at {inside_object_pos}")
            obs, reward, done = pick(env, task, target_pos=inside_object_pos)
            if done:
                print("[Task] Task ended during pick-inside-object.")
                return
            print(f"[Task] Placing object on table at {table_surface_pos}")
            obs, reward, done = place(env, task, target_pos=table_surface_pos)
            if done:
                print("[Task] Task ended during place-on-table.")
                return

        print("===== Task completed successfully =====")

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()