import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

import math

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers if desired
        init_video_writers(obs)

        # Wrap the task for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Expect keys like 'tomato1', 'bin'; adjust to your actual names
        if 'tomato1' not in positions or 'bin' not in positions:
            raise KeyError("Expected keys 'tomato1' and 'bin' in positions dictionary")
        tomato1_pos = positions['tomato1']
        bin_pos     = positions['bin']

        # === Oracle Plan Execution ===

        # 1) Rotate the gripper by 90 degrees around z-axis for a better grasp orientation
        print("[Task] Rotating gripper to 90 degrees around z-axis")
        # Compute quaternion for 90° about z: (x, y, z, w)
        target_quat = [0.0, 0.0, math.sin(math.pi/4), math.cos(math.pi/4)]
        obs, reward, done = rotate(env, task, target_quat=target_quat,
                                   max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Early termination during rotate")
            return

        # 2) Pick the tomato
        print(f"[Task] Picking tomato at position {tomato1_pos}")
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=tomato1_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Task] pick() raised an exception: {e}")
            return
        if done:
            print("[Task] Task ended after pick")
            return

        # 3) Place the tomato into the bin
        print(f"[Task] Placing tomato into bin at position {bin_pos}")
        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=bin_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Task] place() raised an exception: {e}")
            return
        if done:
            print("[Task] Task ended after place")
            return

        # 4) (Optional) Exploration phase: check lock state of a drawer handle if present
        #    This illustrates how to identify a missing predicate (e.g., lock-known)
        if 'drawer_side_pos' in positions and 'drawer_anchor_pos' in positions:
            side_pos   = positions['drawer_side_pos']
            anchor_pos = positions['drawer_anchor_pos']
            print("[Task] Approaching drawer handle (side)")
            obs, reward, done = move(env, task, from_room=None, to_room=None)  # replace with actual args
            # ... exploration code here to pick-drawer and pull, then infer lock state ...
            # e.g. obs, reward, done = pull(env, task, target_obj='drawer', target_loc=anchor_pos)
            # infer lock-known from the outcome
        else:
            print("[Task] No drawer positions found, skipping exploration of lock predicate")

    except Exception as e:
        print(f"[Task] Unhandled exception: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()