# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # get a map of named positions/orientations from the scene
        positions = get_object_positions()
        print("[Info] Retrieved object positions/orientations:", positions.keys())

        # define an oracle plan for opening a drawer
        #   1) rotate gripper to face the handle
        #   2) move to side position of drawer
        #   3) move to anchor position (handle)
        #   4) pick the handle
        #   5) pull the drawer open
        # the actual keys in positions[] need to match your object_positions module
        plan = [
            {
                'skill': 'rotate',
                'target_quat': positions['drawer_handle_quat'],
                'max_steps': 100,
                'threshold': 0.05,
                'timeout': 10.0
            },
            {
                'skill': 'move',
                'target_pos': positions['drawer_side_pos'],
                'max_steps': 100,
                'threshold': 0.01,
                'timeout': 10.0
            },
            {
                'skill': 'move',
                'target_pos': positions['drawer_anchor_pos'],
                'max_steps': 100,
                'threshold': 0.01,
                'timeout': 10.0
            },
            {
                'skill': 'pick',
                'target_pos': positions['drawer_handle_pos'],
                'approach_distance': 0.02,
                'max_steps': 100,
                'threshold': 0.005,
                'approach_axis': 'z',
                'timeout': 5.0
            },
            {
                'skill': 'pull',
                # assuming your pull skill needs no extra args or uses defaults
            }
        ]

        # execute each step in the plan
        for step_idx, step in enumerate(plan, start=1):
            skill_name = step.pop('skill')
            skill_fn = globals().get(skill_name)
            if skill_fn is None:
                print(f"[Error] Skill '{skill_name}' is not available.")
                return
            print(f"[Plan] Step {step_idx}: {skill_name} with params {step}")
            try:
                obs, reward, done = skill_fn(env, task, **step)
            except TypeError as e:
                print(f"[Error] wrong arguments for skill '{skill_name}': {e}")
                return
            except Exception as e:
                print(f"[Error] exception during '{skill_name}': {e}")
                return

            if done:
                print(f"[Task] execution terminated early at step {step_idx} ({skill_name}).")
                return

        print("[Task] plan executed successfully, drawer should now be open.")

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()