import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import rotate, move, pick, place, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # get all object positions as a dict: { name: np.array([x,y,z]) }
        positions = get_object_positions()

        # 1) Slide open any available drawer
        # find a drawer in the scene
        drawer_key = next((k for k in positions.keys() if "drawer" in k), None)
        if drawer_key is None:
            print("[Task] No drawer object found in positions; aborting.")
            return

        drawer_pos = positions[drawer_key]
        print(f"[Task] Found drawer '{drawer_key}' at {drawer_pos}")

        # rotate gripper to ninety_deg orientation before approaching handle
        # quaternion for 90° around Z axis: [0,0,sin(π/4),cos(π/4)]
        ninety_deg_quat = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]
        try:
            obs, reward, done = rotate(env, task, ninety_deg_quat,
                                       max_steps=100, threshold=0.05, timeout=10.0)
        except Exception as e:
            print(f"[Error] rotate failed: {e}")
            return
        if done:
            print("[Task] Terminated during rotate.")
            return

        # pick the drawer handle (using pick-drawer skill)
        print(f"[Task] Picking drawer '{drawer_key}'")
        try:
            obs, reward, done = pick(env, task,
                                     target_pos=drawer_pos,
                                     approach_distance=0.05,
                                     max_steps=100,
                                     threshold=0.01,
                                     approach_axis='z',
                                     timeout=5.0)
        except Exception as e:
            print(f"[Error] pick (drawer) failed: {e}")
            return
        if done:
            print("[Task] Terminated during pick-drawer.")
            return

        # pull the drawer open
        print(f"[Task] Pulling drawer '{drawer_key}'")
        try:
            obs, reward, done = pull(env, task,
                                     target_pos=drawer_pos,
                                     max_steps=150,
                                     threshold=0.01,
                                     timeout=10.0)
        except Exception as e:
            print(f"[Error] pull failed: {e}")
            return
        if done:
            print("[Task] Terminated during pull.")
            return

        print(f"[Task] Drawer '{drawer_key}' should now be open.")

        # 2) Chuck away any rubbish remaining on the table
        # define possible rubbish keys by filtering object names
        rubbish_keys = [k for k in positions.keys()
                        if k != drawer_key and ("trash" in k or "rubbish" in k or "paper" in k)]
        if not rubbish_keys:
            print("[Task] No rubbish objects identified on table.")
        else:
            # find a disposal location or fallback
            bin_key = next((k for k in positions.keys() if "bin" in k or "disposal" in k), None)
            if bin_key:
                disposal_pos = positions[bin_key]
                print(f"[Task] Using disposal '{bin_key}' at {disposal_pos}")
            else:
                # fallback disposal position: just in front of robot at origin offset
                disposal_pos = np.array([0.5, 0.0, 0.2])
                print(f"[Task] No bin found, using fallback disposal at {disposal_pos}")

            for obj_key in rubbish_keys:
                obj_pos = positions[obj_key]
                print(f"[Task] Picking up rubbish '{obj_key}' at {obj_pos}")
                try:
                    obs, reward, done = pick(env, task,
                                             target_pos=obj_pos,
                                             approach_distance=0.10,
                                             max_steps=100,
                                             threshold=0.01,
                                             approach_axis='z',
                                             timeout=5.0)
                except Exception as e:
                    print(f"[Error] pick (rubbish '{obj_key}') failed: {e}")
                    return
                if done:
                    print("[Task] Terminated during pick of rubbish.")
                    return

                print(f"[Task] Placing rubbish '{obj_key}' at disposal")
                try:
                    obs, reward, done = place(env, task,
                                               target_pos=disposal_pos,
                                               approach_distance=0.10,
                                               max_steps=100,
                                               threshold=0.01,
                                               approach_axis='z',
                                               timeout=5.0)
                except Exception as e:
                    print(f"[Error] place (rubbish '{obj_key}') failed: {e}")
                    return
                if done:
                    print("[Task] Terminated during place of rubbish.")
                    return

        print("[Task] Completed drawer opening and rubbish disposal.")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()