import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset and initialize recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve positions of relevant objects
        positions = get_object_positions()
        # We assume the keys contain 'drawer_side_pos' and 'drawer_anchor_pos'
        if 'drawer_side_pos' not in positions or 'drawer_anchor_pos' not in positions:
            raise KeyError("Expected keys 'drawer_side_pos' and 'drawer_anchor_pos' in positions")
        side_pos = np.array(positions['drawer_side_pos'])
        anchor_pos = np.array(positions['drawer_anchor_pos'])

        # Names of the drawer and gripper as known to the environment
        drawer_name = 'drawer'   # replace with actual drawer name if different
        gripper_name = 'gripper' # replace with actual gripper name if different

        # == Exploration Phase: Determine if gripper is already rotated ==
        obs = task.get_observation()
        # gripper_pose: [x, y, z, qx, qy, qz, qw]
        quat = obs.gripper_pose[3:7]
        # euler_from_quat returns (roll, pitch, yaw)
        r, p, yaw = euler_from_quat(quat)
        print(f"[Exploration] Current gripper yaw: {yaw}")
        # If not at ninety degrees, rotate
        # We define 'zero_deg' and 'ninety_deg' as the symbolic angles in the domain
        if abs(yaw) < 1e-2:
            print("[Task] Rotating gripper from zero_deg to ninety_deg")
            try:
                obs, reward, done = rotate(env, task, gripper_name, 'zero_deg', 'ninety_deg')
            except Exception as e:
                print(f"[Error] rotate failed: {e}")
                return
            if done:
                print("[Task] Task ended after rotate")
                return

        # == Move phase: Approach the side position of the drawer ==
        print("[Task] Moving gripper to side position of the drawer")
        try:
            obs, reward, done = move(env, task, side_pos)
        except Exception as e:
            print(f"[Error] move to side_pos failed: {e}")
            return
        if done:
            print("[Task] Task ended after move to side")
            return

        # == Move phase: Approach the anchor position (handle) of the drawer ==
        print("[Task] Moving gripper to anchor position of the drawer")
        try:
            obs, reward, done = move(env, task, anchor_pos)
        except Exception as e:
            print(f"[Error] move to anchor_pos failed: {e}")
            return
        if done:
            print("[Task] Task ended after move to anchor")
            return

        # == Grasp the drawer handle ==
        print("[Task] Picking (grasping) the drawer handle")
        # approach along x-axis, keep defaults for other parameters
        try:
            obs, reward, done = pick(env,
                                     task,
                                     target_pos=anchor_pos,
                                     approach_distance=0.10,
                                     max_steps=100,
                                     threshold=0.01,
                                     approach_axis='x',
                                     timeout=10.0)
        except Exception as e:
            print(f"[Error] pick failed: {e}")
            return
        if done:
            print("[Task] Task ended after pick")
            return

        # == Pull to open the drawer ==
        print("[Task] Pulling to open the drawer")
        try:
            obs, reward, done = pull(env, task)
        except Exception as e:
            print(f"[Error] pull failed: {e}")
            return
        if done:
            print("[Task] Task ended after pull")
            return

        print("===== Task plan executed successfully =====")

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()