import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)
        
        # get named object positions from the scene
        positions = get_object_positions()
        # keys should include 'drawer_handle', 'drawer_side_pos', 'drawer_anchor_pos'
        drawer_handle_pos = positions['drawer_handle']
        drawer_side_pos   = positions['drawer_side_pos']
        drawer_anchor_pos = positions['drawer_anchor_pos']
        
        # 1) Rotate gripper to the side orientation required for approaching the drawer side
        ninety_deg_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])
        print("[Task] Rotating gripper to ninety degrees")
        obs, reward, done = rotate(env, task, ninety_deg_quat)
        if done:
            print("[Task] Task ended during rotation")
            return
        
        # 2) Move to a safe side position of the drawer
        print("[Task] Moving to drawer side position:", drawer_side_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=drawer_side_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended during move-to-side")
            return
        
        # 3) Move from side to anchor position on the drawer handle
        print("[Task] Moving to anchor position on handle:", drawer_anchor_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=drawer_anchor_pos,
            approach_distance=0.05,
            max_steps=100,
            threshold=0.005,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended during move-to-anchor")
            return
        
        # 4) Pick (grasp) the drawer handle
        print("[Task] Grasping drawer handle")
        obs, reward, done = pick(
            env,
            task,
            target_pos=drawer_anchor_pos,
            approach_distance=0.02,
            max_steps=100,
            threshold=0.005,
            approach_axis='z',
            timeout=5.0
        )
        if done:
            print("[Task] Task ended during pick-drawer-handle")
            return
        
        # 5) Pull the drawer to open or to inspect lock state
        print("[Task] Pulling drawer handle")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Task ended during pull")
            return
        
        print("[Task] Drawer should now be open (or lock state identified).")
    
    finally:
        shutdown_environment(env)
    
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()