import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running a drawer‐opening task and discovering a missing predicate.'''
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset the task
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # get known object positions
        positions = get_object_positions()
        # Keys we expect in positions:
        #   'drawer_side_pos': a 3‐vector for the gripper side position
        #   'drawer_anchor_pos': a 3‐vector for the gripper anchor position
        #   'drawer_handle': string or object name for the drawer handle

        if 'drawer_side_pos' not in positions or 'drawer_anchor_pos' not in positions:
            raise KeyError("Expected keys 'drawer_side_pos' and 'drawer_anchor_pos' in object_positions")

        side_pos = positions['drawer_side_pos']
        anchor_pos = positions['drawer_anchor_pos']

        # Step 1: Move to the side of the drawer
        print("[Task] Moving gripper to side position:", side_pos)
        obs, reward, done = move(
            env, task,
            target_pos=side_pos,
            max_steps=200,
            threshold=0.02,
            approach_axis='z',
            timeout=15.0
        )
        if done:
            print("[Task] Ended after move to side!")
            return

        # Step 2: Rotate to ninety degrees (open orientation)
        # We define a quaternion corresponding roughly to a 90° rotation about the X axis
        ninety_deg_quat = np.array([np.cos(np.pi/4), np.sin(np.pi/4), 0.0, 0.0])
        print("[Task] Rotating gripper to ninety degrees orientation:", ninety_deg_quat)
        obs, reward, done = rotate(
            env, task,
            target_quat=ninety_deg_quat,
            max_steps=150,
            threshold=0.02,
            timeout=10.0
        )
        if done:
            print("[Task] Ended during rotation!")
            return

        # Step 3: Move from side position to anchor position on the drawer handle
        print("[Task] Moving gripper from side to anchor on drawer:", anchor_pos)
        obs, reward, done = move(
            env, task,
            target_pos=anchor_pos,
            max_steps=200,
            threshold=0.01,
            approach_axis='x',
            timeout=15.0
        )
        if done:
            print("[Task] Ended after move to anchor!")
            return

        # Step 4: Attempt to pull the drawer open
        print("[Task] Pulling drawer handle to open the drawer")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Task ended during pull!")
            return

        # If pull fails or drawer remains closed, we may be missing a 'lock-known' predicate.
        # Log for exploration:
        is_open = getattr(obs, 'drawer_open', None)
        if not is_open:
            print("[Exploration] Drawer did not open. We suspect a missing 'lock-known' predicate.")
            # In a real exploration phase, we would now acquire 'lock-known' by some sensing or action.

        print("[Task] Completed drawer-opening sequence.")

    except Exception as e:
        print("[Error] Exception during task execution:", e)
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()