import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # retrieve positions
        positions = get_object_positions()
        print(f"[Info] Retrieved object positions: {list(positions.keys())}")

        # === Exploration phase ===
        print("[Exploration] Starting exploration phase")
        for name, pos in positions.items():
            try:
                print(f"[Exploration] Moving to {name} at {pos}")
                obs, reward, done = move(env, task, pos)
                if done:
                    print(f"[Exploration] Task ended during exploration at {name}")
                    return
                print(f"[Exploration] Identified {name}")
            except Exception as e:
                print(f"[Exploration] Error exploring {name}: {e}")
        print("[Exploration] Completed exploration phase")

        # find drawer handle positions
        side_keys = [k for k in positions if 'side' in k]
        anchor_keys = [k for k in positions if 'anchor' in k]
        if not side_keys or not anchor_keys:
            print("[Error] Could not find drawer handle positions in retrieved keys")
            return
        drawer_side_pos = positions[side_keys[0]]
        drawer_anchor_pos = positions[anchor_keys[0]]

        # safety check: do not proceed if drawer locked
        if hasattr(obs, 'is_locked') and obs.is_locked:
            print("Warning: Drawer is locked. Aborting plan to avoid damage")
            return

        # === Main plan ===
        print("[Plan] Rotating gripper to open orientation")
        # quaternion for ~90 degree rotation about z (as example)
        target_quat = [0.0, 0.0, 0.7071, 0.7071]
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("Task ended during rotation")
            return

        print("[Plan] Moving to drawer side position")
        obs, reward, done = move(env, task, drawer_side_pos)
        if done:
            print("Task ended during move to side")
            return

        print("[Plan] Moving to drawer anchor position")
        obs, reward, done = move(env, task, drawer_anchor_pos)
        if done:
            print("Task ended during move to anchor")
            return

        print("[Plan] Picking drawer handle")
        obs, reward, done = pick(env, task, drawer_anchor_pos)
        if done:
            print("Task ended during pick drawer handle")
            return

        print("[Plan] Pulling drawer open")
        obs, reward, done = pull(env, task)
        if done:
            print("Task ended during pull")
            return

        print("===== Task plan executed successfully =====")

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()