# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        print("[Info] Retrieved object positions:", positions.keys())

        # Identify drawer-related positions by key names
        # Expect keys like 'drawer_side', 'drawer_anchor', 'disposal_bin', etc.
        # Fallback if names differ
        side_key = next((k for k in positions if 'side' in k), None)
        anchor_key = next((k for k in positions if 'anchor' in k), None)
        bin_key = next((k for k in positions if 'bin' in k or 'disposal' in k), None)

        if side_key is None or anchor_key is None or bin_key is None:
            raise KeyError("Could not find expected position keys in object_positions.")

        drawer_side_pos = np.array(positions[side_key])
        drawer_anchor_pos = np.array(positions[anchor_key])
        disposal_bin_pos = np.array(positions[bin_key])

        print(f"[Task] Side position: {side_key} -> {drawer_side_pos}")
        print(f"[Task] Anchor position: {anchor_key} -> {drawer_anchor_pos}")
        print(f"[Task] Disposal bin position: {bin_key} -> {disposal_bin_pos}")

        # === Oracle Plan Execution ===

        # 1) Move gripper to the side position of the drawer
        obs, reward, done = move(env, task, target_pos=drawer_side_pos)
        if done:
            print("[Task] Episode ended unexpectedly after move to side.")
            return

        # 2) Rotate gripper to ninety degrees to align for opening
        #    Assume rotate signature: rotate(env, task, from_angle, to_angle)
        print("[Task] Rotating gripper to ninety_deg")
        obs, reward, done = rotate(env, task, 'zero_deg', 'ninety_deg')
        if done:
            print("[Task] Episode ended unexpectedly during rotate.")
            return

        # 3) Move gripper from side position to anchor position on the drawer
        obs, reward, done = move(env, task, target_pos=drawer_anchor_pos)
        if done:
            print("[Task] Episode ended unexpectedly after move to anchor.")
            return

        # 4) Pick the drawer handle at the anchor position
        #    We use the generic pick skill; assume it grasps whatever is at target_pos
        print("[Task] Grasping the drawer handle")
        obs, reward, done = pick(env, task, target_pos=drawer_anchor_pos)
        if done:
            print("[Task] Episode ended unexpectedly during pick.")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling drawer open")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Episode ended unexpectedly during pull.")
            return

        # 6) Move the gripper (with any held object) to the disposal bin
        obs, reward, done = move(env, task, target_pos=disposal_bin_pos)
        if done:
            print("[Task] Episode ended unexpectedly before place.")
            return

        # 7) Place any carried object into the disposal bin
        print("[Task] Placing object into disposal bin")
        obs, reward, done = place(env, task, target_pos=disposal_bin_pos)
        if done:
            print("[Task] Episode ended after place.")
            return

        print("[Task] Oracle plan completed successfully.")

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()