import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve all known object positions
        positions = get_object_positions()
        # Identify side‐of‐drawer and anchor positions by key name
        side_key = next(k for k in positions if 'side' in k.lower())
        anchor_key = next(k for k in positions if 'anchor' in k.lower())
        side_pos = positions[side_key]
        anchor_pos = positions[anchor_key]

        # Define quaternions for 0° and 90° rotations around gripper Z‐axis
        # (assuming [x, y, z, w] order)
        zero_deg_quat = [0.0, 0.0, 0.0, 1.0]
        ninety_deg_quat = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]

        # 1) Ensure gripper is at zero orientation
        print("[Task] Resetting gripper orientation to 0°")
        obs, reward, done = rotate(env, task, zero_deg_quat)
        if done:
            print("[Task] Terminated early after zero‐deg rotate")
            return

        # 2) Rotate gripper to 90° to satisfy 'move‐to‐side' precondition
        print("[Task] Rotating gripper to 90° for side approach")
        obs, reward, done = rotate(env, task, ninety_deg_quat)
        if done:
            print("[Task] Terminated early after 90° rotate")
            return

        # 3) Approach the side of the drawer
        print(f"[Task] Moving to side position of drawer at {side_pos}")
        obs, reward, done = move(
            env, task,
            target_pos=side_pos,
            approach_distance=0.05,
            max_steps=200,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Terminated early during side approach")
            return

        # 4) Approach the anchor (handle) of the drawer
        print(f"[Task] Moving to anchor (handle) position of drawer at {anchor_pos}")
        obs, reward, done = move(
            env, task,
            target_pos=anchor_pos,
            approach_distance=0.03,
            max_steps=200,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Terminated early during anchor approach")
            return

        # 5) Grasp the drawer handle (pick-drawer)
        print("[Task] Grasping drawer handle")
        obs, reward, done = pick(
            env, task,
            target_pos=anchor_pos,
            approach_distance=0.02,
            max_steps=200,
            threshold=0.005,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Terminated early during handle grasp")
            return

        # 6) Pull the drawer open
        print("[Task] Pulling the drawer open")
        obs, reward, done = pull(
            env, task,
            target_pos=anchor_pos,
            approach_distance=0.1,
            max_steps=200,
            threshold=0.01,
            approach_axis='x',
            timeout=10.0
        )
        if done:
            print("[Task] Terminated early during drawer pull")
            return

        print("[Task] Successfully opened the drawer!")
    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()