import numpy as np
from env import setup_environment, shutdown_environment
from skill_code import pick, place, move, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        # This returns a dict mapping object names to 3‐tuples or numpy arrays
        positions = get_object_positions()
        print("[Info] Retrieved object positions:", positions)

        # Validate that our key objects exist
        required_objects = ['rubbish', 'bin']
        for obj in required_objects:
            if obj not in positions:
                raise KeyError(f"Object '{obj}' not found in positions dictionary")

        rubbish_pos = np.array(positions['rubbish'])
        bin_pos      = np.array(positions['bin'])

        # === Exploration Phase ===
        # Move to rubbish to identify and sense it
        print("[Exploration] Moving to rubbish location for identification and sensing")
        try:
            obs, reward, done = move(env, task,
                from_position=None,  # unspecified, assume move can infer current
                to_position=rubbish_pos)
        except Exception as e:
            print(f"[Error] Exploration move failed: {e}")
            return
        if done:
            print("[Info] Task ended during exploration move")
            return

        # Optionally pick and place back to sense weight/durability
        print("[Exploration] Grasping rubbish briefly to measure weight/durability")
        try:
            obs, reward, done = pick(env, task,
                target_pos=rubbish_pos,
                approach_distance=0.10,
                max_steps=50,
                threshold=0.01,
                approach_axis='z',
                timeout=5.0)
        except Exception as e:
            print(f"[Error] Exploration pick failed: {e}")
            return
        if done:
            print("[Info] Task ended after exploration pick")
            return

        # Put it back
        print("[Exploration] Placing rubbish back to original location")
        try:
            obs, reward, done = place(env, task,
                target_pos=rubbish_pos,
                approach_distance=0.10,
                max_steps=50,
                threshold=0.01,
                approach_axis='z',
                timeout=5.0)
        except Exception as e:
            print(f"[Error] Exploration place failed: {e}")
            return
        if done:
            print("[Info] Task ended after exploration place")
            return

        # === Main Task: Pick up rubbish and deposit in bin ===

        # 1) Check if rubbish is already in bin
        dist_rubbish_to_bin = np.linalg.norm(rubbish_pos - bin_pos)
        if dist_rubbish_to_bin < 0.05:
            print("[Task] Rubbish already in bin. No action required.")
            return

        # 2) Approach and pick up the rubbish
        print("[Task] Picking up rubbish at:", rubbish_pos)
        try:
            obs, reward, done = pick(env, task,
                target_pos=rubbish_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0)
        except Exception as e:
            print(f"[Error] pick action failed: {e}")
            return
        if done:
            print("[Task] Task ended after pick")
            return

        # 3) Move to bin location (optional: use move if available)
        print("[Task] Moving towards bin position for placement")
        try:
            obs, reward, done = move(env, task,
                from_position=None,
                to_position=bin_pos)
        except Exception as e:
            print(f"[Error] move to bin failed: {e}")
            return
        if done:
            print("[Task] Task ended during move to bin")
            return

        # 4) Place the rubbish in the bin
        print("[Task] Placing rubbish in bin at:", bin_pos)
        try:
            obs, reward, done = place(env, task,
                target_pos=bin_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0)
        except Exception as e:
            print(f"[Error] place action failed: {e}")
            return
        if done:
            print("[Task] Task ended after place")
            return

        print("[Task] Successfully disposed of the rubbish.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()