import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Expect keys like 'object_1', 'object_2', 'drawer_side', 'drawer_anchor', 'robot_start'

        # === Exploration Phase: Identify Object Properties ===
        print("[Exploration] Identifying weight and durability of objects")
        for name, pos in positions.items():
            if name.startswith('object_'):
                print(f"[Exploration] Moving to and grasping {name} at {pos}")
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=pos,
                    approach_distance=0.1,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=5.0
                )
                if done:
                    print("[Exploration] Task ended unexpectedly during pick!")
                    return
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=pos,
                    place_height=pos[2] + 0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=5.0
                )
                if done:
                    print("[Exploration] Task ended unexpectedly during place!")
                    return

        # === Exploration Phase: Discover Drawer Lock Predicate ===
        drawer_side = positions.get('drawer_side')
        drawer_anchor = positions.get('drawer_anchor')
        print("[Exploration] Testing drawer lock state with pull action")
        # Step 1: Rotate gripper to ninety_deg orientation
        ninety_quat = [0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)]
        obs, reward, done = rotate(env, task, target_quat=ninety_quat,
                                   max_steps=100, threshold=0.05, timeout=10.0)
        if done:
            print("[Exploration] Task ended during rotation!")
            return

        # Step 2: Approach the drawer handle and grasp it
        obs, reward, done = pick(
            env,
            task,
            target_pos=drawer_anchor,
            approach_distance=0.02,
            max_steps=100,
            threshold=0.005,
            approach_axis='x',
            timeout=5.0
        )
        if done:
            print("[Exploration] Task ended during drawer grasp!")
            return

        # Step 3: Pull to determine lock state
        obs, reward, done = pull(env, task)
        if done:
            print("[Exploration] Task ended during pull!")
            return

        # Inspect observation for lock information
        lock_known = getattr(obs, 'lock_known', None)
        print(f"[Exploration] lock-known = {lock_known}")
        print("[Exploration] Identified missing predicate: lock-known")

        # === Oracle Plan: Open the Drawer ===
        print("[Plan] Executing open-drawer sequence")
        # (1) Ensure gripper is at ninety-degree orientation
        obs, reward, done = rotate(env, task, target_quat=ninety_quat,
                                   max_steps=100, threshold=0.05, timeout=10.0)
        if done:
            print("[Plan] Task ended during rotation!")
            return

        # (2) Move to side position (we treat picking the side as approach)
        obs, reward, done = pick(
            env,
            task,
            target_pos=drawer_side,
            approach_distance=0.02,
            max_steps=100,
            threshold=0.005,
            approach_axis='x',
            timeout=5.0
        )
        if done:
            print("[Plan] Task ended during move-to-side step!")
            return

        # (3) Move to anchor position (we treat this as another grasp step)
        obs, reward, done = pick(
            env,
            task,
            target_pos=drawer_anchor,
            approach_distance=0.02,
            max_steps=100,
            threshold=0.005,
            approach_axis='x',
            timeout=5.0
        )
        if done:
            print("[Plan] Task ended during move-to-anchor step!")
            return

        # (4) Pull the drawer open
        obs, reward, done = pull(env, task)
        if done:
            print("[Plan] Task ended during final pull!")
            return

        print("[Plan] Drawer open sequence completed successfully.")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()