import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor
from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for recording (optional)
        init_video_writers(obs)

        # Wrap step and get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve all known object and reference positions / quaternions
        positions = get_object_positions()

        # Replace the placeholder keys with your actual environment labels:
        # These keys must match what get_object_positions() returns.
        # For a drawer‐opening task, you typically have:
        #   'drawer_handle_anchor' : the 3D pos of the handle you pull
        #   'drawer_handle_side'   : a side approach position
        #   'gripper_quat_zero'    : quaternion at zero_deg orientation
        #   'gripper_quat_ninety'  : quaternion at ninety_deg orientation
        anchor_pos = positions['drawer_handle_anchor']
        side_pos   = positions['drawer_handle_side']
        quat_zero  = positions['gripper_quat_zero']
        quat_ninety= positions['gripper_quat_ninety']

        # 1) Ensure gripper starts at zero_deg orientation if needed
        print("[Task] Rotating to zero_deg orientation")
        obs, reward, done = rotate(env, task, target_quat=quat_zero)
        if done:
            print("[Task] Early termination after zero_deg rotate")
            return

        # 2) Rotate from zero_deg to ninety_deg for side approach
        print("[Task] Rotating to ninety_deg orientation")
        obs, reward, done = rotate(env, task, target_quat=quat_ninety)
        if done:
            print("[Task] Early termination after ninety_deg rotate")
            return

        # 3) Move the gripper to a side‐position of the drawer
        print(f"[Task] Moving to side position: {side_pos}")
        obs, reward, done = move(
            env,
            task,
            target_pos=side_pos,
            approach_distance=0.10,
            max_steps=100,
            threshold=0.005,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Early termination after move to side")
            return

        # 4) Move from side‐position to the anchor/handle of the drawer
        print(f"[Task] Moving to anchor position: {anchor_pos}")
        obs, reward, done = move(
            env,
            task,
            target_pos=anchor_pos,
            approach_distance=0.02,
            max_steps=100,
            threshold=0.005,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Early termination after move to anchor")
            return

        # 5) Grasp the drawer handle (pick‐drawer)
        print("[Task] Picking drawer handle")
        obs, reward, done = pick(
            env,
            task,
            target_pos=anchor_pos,
            approach_distance=0.02,
            max_steps=100,
            threshold=0.005,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Early termination after pick")
            return

        # 6) Pull the drawer open
        print("[Task] Pulling drawer open")
        obs, reward, done = pull(
            env,
            task,
            target_pos=anchor_pos,
            pull_distance=0.15,
            max_steps=200,
            threshold=0.01,
            timeout=15.0
        )
        if done:
            print("[Task] Early termination after pull")
            return

        print("===== Task completed successfully =====")

    except Exception as e:
        print(f"[Task] Exception encountered: {e}")
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
