import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")

    env, task = setup_environment()
    try:
        # Reset and initialize video recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve object positions from the scene
        positions = get_object_positions()
        # We expect these keys in the positions dict
        # 'drawer_side_pos', 'drawer_anchor_pos'
        if 'drawer_side_pos' not in positions or 'drawer_anchor_pos' not in positions:
            print("[Error] Required drawer positions not found in get_object_positions() output.")
            return

        # 1) Rotate gripper to zero_deg orientation (identity quaternion)
        zero_quat = [0.0, 0.0, 0.0, 1.0]
        print("[Task] Rotating gripper to zero_deg orientation")
        obs, reward, done = rotate(env, task, zero_quat)
        if done:
            print("[Task] Task ended during rotate!")
            return

        # 2) Move gripper to the side of the drawer
        side_pos = positions['drawer_side_pos']
        print(f"[Task] Moving gripper to side position: {side_pos}")
        obs, reward, done = move(
            env, task,
            target_pos=side_pos,
            approach_distance=0.05,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended during move-to-side!")
            return

        # 3) Move gripper to the anchor (handle) position of the drawer
        anchor_pos = positions['drawer_anchor_pos']
        print(f"[Task] Moving gripper to anchor position: {anchor_pos}")
        obs, reward, done = move(
            env, task,
            target_pos=anchor_pos,
            approach_distance=0.02,
            max_steps=100,
            threshold=0.005,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended during move-to-anchor!")
            return

        # 4) Pick (grasp) the drawer handle
        print("[Task] Picking the drawer handle")
        obs, reward, done = pick(
            env, task,
            target_pos=anchor_pos,
            approach_distance=0.0,
            max_steps=50,
            threshold=0.01,
            approach_axis='z',
            timeout=5.0
        )
        if done:
            print("[Task] Task ended during pick-drawer!")
            return

        # 5) Pull the drawer open
        print("[Task] Pulling the drawer open")
        # We assume pull takes a distance and speed parameter
        obs, reward, done = pull(
            env, task,
            distance=0.2,
            speed=0.02,
            max_steps=100,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended during pull!")
            return

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
