import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import rotate, pick, place, move, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    env, task = setup_environment()
    try:
        # Reset to initial state
        descriptions, obs = task.reset()

        # Initialize video recording if desired
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve known object world‐frame positions
        positions = get_object_positions()
        print(f"[Info] Retrieved object positions: {list(positions.keys())}")

        # === Exploration Phase: Identify missing predicate ===
        obs = task.get_observation()
        # collect all observable predicate‐like attributes
        attrs = [a for a in dir(obs) if not a.startswith('_')]
        print(f"[Exploration] Observation available attributes: {attrs}")
        if 'rotated' not in attrs:
            print("[Exploration] Missing predicate detected: 'rotated'")

        # === Example Plan Execution ===
        # For example, rotate the gripper to ninety_deg orientation,
        # then perform subsequent actions as needed.
        # Define a target quaternion corresponding to a 90° rotation
        # about the gripper’s local axis (example values).
        ninety_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])
        print("[Plan] Rotating gripper to ninety degrees.")
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("[Plan] Terminated during rotate step.")
            return

        # Placeholder for further plan steps using pick/place/move/pull
        # Example:
        # obj_name = 'target_object'
        # obj_pos = positions.get(obj_name)
        # if obj_pos is None:
        #     raise KeyError(f"Object position for '{obj_name}' not found.")
        # print(f"[Plan] Picking up {obj_name} at {obj_pos}.")
        # obs, reward, done = pick(env, task, obj_name, obj_pos, 'gripper')
        # if done:
        #     print("[Plan] Task ended during pick step.")
        #     return

        # obs, reward, done = pull(env, task, 'gripper', 'drawer')
        # if done:
        #     print("[Plan] Task ended during pull step.")
        #     return

        print("[Plan] Skeleton plan executed (rotate only). Extend this with pick/place/move/pull as needed.")

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()