import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)
        
        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # we expect the dictionary to contain keys for the drawer handle side and anchor
        # these correspond to the PDDL predicates is-side-pos and is-anchor-pos
        side_pos = positions['drawer_handle_side']
        anchor_pos = positions['drawer_handle_anchor']
        
        # === Oracle Plan Execution ===
        # 1) Move to the side of the drawer
        print("[Task] Moving gripper to drawer side position:", side_pos)
        obs, reward, done = move(env, task, target_pos=np.array(side_pos))
        if done:
            print("[Task] Ended early after move to side.")
            return
        
        # 2) Rotate gripper so it can approach the drawer anchor
        #    We assume initial angle is "zero_deg" and we want "ninety_deg"
        print("[Task] Rotating gripper to ninety degrees.")
        obs, reward, done = rotate(env, task, from_angle='zero_deg', to_angle='ninety_deg')
        if done:
            print("[Task] Ended early after rotate.")
            return
        
        # 3) Move to the anchor position of the drawer
        print("[Task] Moving gripper to drawer anchor position:", anchor_pos)
        obs, reward, done = move(env, task, target_pos=np.array(anchor_pos))
        if done:
            print("[Task] Ended early after move to anchor.")
            return
        
        # 4) Pick the drawer handle at the anchor position
        print("[Task] Picking up drawer handle.")
        obs, reward, done = pick(env, task, target_pos=np.array(anchor_pos))
        if done:
            print("[Task] Ended early after pick.")
            return
        
        # 5) Pull the drawer open
        print("[Task] Pulling drawer open.")
        obs, reward, done = pull(env, task)
        if done:
            print("[Task] Ended after pull.")
            return
        
        print("[Task] Drawer should now be open. Plan completed.")
        
    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()