# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # It is assumed get_object_positions returns a dict of names to (x,y,z) tuples
        positions = get_object_positions()

        # Identify the drawer side and anchor positions in the scene
        side_keys = [k for k in positions.keys() if 'side' in k]
        anchor_keys = [k for k in positions.keys() if 'anchor' in k]
        if not side_keys or not anchor_keys:
            raise RuntimeError("Cannot find 'side' or 'anchor' position in get_object_positions output.")
        side_pos = np.array(positions[side_keys[0]])
        anchor_pos = np.array(positions[anchor_keys[0]])

        # === Oracle Plan ===
        # 1) Move gripper to side position
        print("[Plan] Moving to drawer side position:", side_pos)
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Plan] Task ended prematurely during move to side.")
            return

        # 2) Rotate gripper to ninety_deg
        #    We assume the current angle label is 'zero_deg'; adjust if different
        print("[Plan] Rotating gripper from zero_deg to ninety_deg")
        try:
            obs, reward, done = rotate(env, task, 'zero_deg', 'ninety_deg')
        except Exception as e:
            print(f"[Plan] Rotation failed or not needed: {e}")
        if done:
            print("[Plan] Task ended prematurely during rotate.")
            return

        # 3) Move from side position to anchor position
        print("[Plan] Moving to drawer anchor position:", anchor_pos)
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Plan] Task ended prematurely during move to anchor.")
            return

        # 4) Pick the drawer handle at the anchor position
        print("[Plan] Picking drawer handle at anchor position.")
        obs, reward, done = pick(env, task, target_pos=anchor_pos,
                                 approach_distance=0.05,
                                 max_steps=100,
                                 threshold=0.01,
                                 approach_axis='z',
                                 timeout=10.0)
        if done:
            print("[Plan] Task ended prematurely during pick-drawer.")
            return

        # 5) Pull the drawer open
        print("[Plan] Pulling drawer to open it.")
        obs, reward, done = pull(env, task,
                                 target_pos=None,
                                 max_steps=100,
                                 threshold=0.01,
                                 timeout=10.0)
        if done:
            print("[Plan] Task ended prematurely during pull.")
            return

        print("[Plan] Drawer should now be open. Task completed.")
        
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()