import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers if desired
        init_video_writers(obs)

        # Wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        try:
            side_pos_bottom = np.array(positions['side-pos-bottom'])
            anchor_pos_bottom = np.array(positions['anchor-pos-bottom'])
            item1_pos = np.array(positions.get('item1', positions.get('tomato1')))
            item2_pos = np.array(positions.get('item2', positions.get('tomato2')))
            plate_pos = np.array(positions['plate'])
        except KeyError as e:
            print(f"[Error] Missing position for object: {e}")
            return

        # Step 1: Move to side-pos-bottom
        print("[Task] Step 1: Moving to side-pos-bottom")
        obs, reward, done = move(env, task, target_pos=side_pos_bottom)
        if done:
            print("[Task] Episode ended early after move to side-pos-bottom.")
            return

        # Step 2: Rotate gripper to ninety_deg
        print("[Task] Step 2: Rotating to ninety_deg")
        angle_shape = Shape('ninety_deg')
        target_quat = np.array(angle_shape.get_quaternion())
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Episode ended early after rotate.")
            return

        # Step 3: Move to anchor-pos-bottom
        print("[Task] Step 3: Moving to anchor-pos-bottom")
        obs, reward, done = move(env, task, target_pos=anchor_pos_bottom)
        if done:
            print("[Task] Episode ended early after move to anchor-pos-bottom.")
            return

        # Step 4: Pull the drawer open
        print("[Task] Step 4: Pulling the drawer open")
        # Choose a pull distance that opens the drawer; adjust if necessary
        pull_distance = 0.1
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
        if done:
            print("[Task] Episode ended early after pull.")
            return

        # Step 5: Pick up the first tomato (item1)
        print("[Task] Step 5: Picking up item1")
        obs, reward, done = pick(env, task, target_pos=item1_pos)
        if done:
            print("[Task] Episode ended early after picking item1.")
            return

        # Step 6: Place item1 onto the plate
        print("[Task] Step 6: Placing item1 onto the plate")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Episode ended after placing item1. Reward:", reward)
            return

        # Step 7: Pick up the second tomato (item2)
        print("[Task] Step 7: Picking up item2")
        obs, reward, done = pick(env, task, target_pos=item2_pos)
        if done:
            print("[Task] Episode ended early after picking item2.")
            return

        # Step 8: Place item2 onto the plate
        print("[Task] Step 8: Placing item2 onto the plate")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Episode ended after placing item2. Reward:", reward)
        else:
            print("[Task] Completed all steps. Final reward:", reward)

    except Exception as e:
        print(f"[Exception] {e}")
    finally:
        # Always shut down environment
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
