from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions
import numpy as np

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve all object positions and any orientation quaternions
        positions = get_object_positions()

        # Step 1: Rotate gripper from zero_deg to ninety_deg
        print("[Task] Rotating gripper to ninety_deg")
        ninety_quat = np.array(positions['ninety_deg'])
        obs, reward, done = rotate(env, task, target_quat=ninety_quat)
        if done:
            print("[Task] Finished during rotate.")
            return

        # Step 2: Move gripper beside the bottom drawer (side position)
        print("[Task] Moving to bottom side position")
        side_bottom = np.array(positions['bottom_side_pos'])
        obs, reward, done = move(env, task, target_pos=side_bottom)
        if done:
            print("[Task] Finished during move-to-side.")
            return

        # Step 3: Move gripper to bottom anchor position
        print("[Task] Moving to bottom anchor position")
        anchor_bottom = np.array(positions['bottom_anchor_pos'])
        obs, reward, done = move(env, task, target_pos=anchor_bottom)
        if done:
            print("[Task] Finished during move-to-anchor.")
            return

        # Step 4: Pick the drawer handle (treat as pick)
        print("[Task] Picking drawer handle at anchor pos")
        obs, reward, done = pick(env, task, target_pos=anchor_bottom)
        if done:
            print("[Task] Finished during pick-drawer.")
            return

        # Step 5: Pull the drawer open
        print("[Task] Pulling drawer open")
        # assume 0.1m pull distance along x
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Task] Finished during pull.")
            return

        # Step 6: Pick tomato1 from table
        print("[Task] Picking tomato1 from table")
        tomato1_pos = np.array(positions['tomato1'])
        obs, reward, done = pick(env, task, target_pos=tomato1_pos)
        if done:
            print("[Task] Finished during pick tomato1.")
            return

        # Step 7: Place tomato1 onto plate
        print("[Task] Placing tomato1 onto plate")
        plate_pos = np.array(positions['plate'])
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Finished during place tomato1.")
            return

        # Step 8: Pick tomato2 from table
        print("[Task] Picking tomato2 from table")
        tomato2_pos = np.array(positions['tomato2'])
        obs, reward, done = pick(env, task, target_pos=tomato2_pos)
        if done:
            print("[Task] Finished during pick tomato2.")
            return

        # Step 9: Place tomato2 onto plate
        print("[Task] Placing tomato2 onto plate")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Finished during place tomato2.")
            return

        print("[Task] All steps executed. Final reward:", reward)

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()