import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # We choose the bottom drawer to open
        side_pos = positions['bottom_side_pos']
        anchor_pos = positions['bottom_anchor_pos']
        # Identify tomatoes (here item1 and item2) and plate
        tomato1_name = 'item1'
        tomato2_name = 'item2'
        tomato1_pos = positions[tomato1_name]
        tomato2_pos = positions[tomato2_name]
        plate_pos   = positions['plate']

        # === Task Plan Execution ===

        # Step 1: Rotate gripper to 90 degrees to align with drawer handle
        quat_ninety = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])
        print("[Task] Step 1: Rotating gripper to ninety degrees")
        obs, reward, done = rotate(env, task, quat_ninety)
        if done:
            print("[Task] Early termination after rotate. Reward:", reward)
            return

        # Step 2: Move gripper to side position of drawer
        print("[Task] Step 2: Moving to drawer side position at", side_pos)
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Task] Early termination after move-to-side. Reward:", reward)
            return

        # Step 3: Pull drawer open fully
        # Compute pull distance and axis
        diff = anchor_pos - side_pos
        pull_dist = np.linalg.norm(diff)
        # Determine primary axis for pull
        axis_map = {0: 'x', 1: 'y', 2: 'z'}
        axis_idx = int(np.argmax(np.abs(diff)))
        pull_axis = axis_map[axis_idx]
        if diff[axis_idx] < 0:
            pull_axis = '-' + pull_axis
        print(f"[Task] Step 3: Pulling drawer along {pull_axis} by distance {pull_dist:.3f}")
        obs, reward, done = pull(env, task, pull_dist, pull_axis)
        if done:
            print("[Task] Early termination after pull. Reward:", reward)
            return

        # Step 4: Pick first tomato (item1)
        print(f"[Task] Step 4: Picking {tomato1_name} at position {tomato1_pos}")
        obs, reward, done = pick(env, task, tomato1_pos)
        if done:
            print(f"[Task] Early termination after pick {tomato1_name}. Reward:", reward)
            return

        # Step 5: Place first tomato onto plate
        print(f"[Task] Step 5: Placing {tomato1_name} onto plate at {plate_pos}")
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print(f"[Task] Early termination after place {tomato1_name}. Reward:", reward)
            return

        # Step 6: Pick second tomato (item2)
        print(f"[Task] Step 6: Picking {tomato2_name} at position {tomato2_pos}")
        obs, reward, done = pick(env, task, tomato2_pos)
        if done:
            print(f"[Task] Early termination after pick {tomato2_name}. Reward:", reward)
            return

        # Step 7: Place second tomato onto plate
        print(f"[Task] Step 7: Placing {tomato2_name} onto plate at {plate_pos}")
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print(f"[Task] Early termination after place {tomato2_name}. Reward:", reward)
            return

        # Final check
        if not done:
            print("[Task] Completed all steps. Final reward:", reward)
        else:
            print("[Task] Task signaled done. Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()