import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers if needed
        init_video_writers(obs)

        # Wrap steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve object positions
        positions = get_object_positions()
        tomato1_pos = positions['tomato1']
        tomato2_pos = positions['tomato2']
        plate_pos = positions['plate']

        # Drawer handle approach positions (if provided)
        # These may be present in your environment; if not, pulling from current gripper pose should still work
        # side_pos_bottom = positions.get('side-pos-bottom', None)
        # anchor_pos_bottom = positions.get('anchor-pos-bottom', None)

        # === Oracle Plan Execution ===
        # 1) Rotate gripper from zero to ninety degrees about its local z-axis
        print("[Plan] Step 1: rotate gripper to 90 degrees about Z")
        # quaternion for 90° about z = [0,0,sin(45°),cos(45°)]
        qz = np.sin(np.pi/4)
        qw = np.cos(np.pi/4)
        target_quat = np.array([0.0, 0.0, qz, qw])
        obs, reward, done = rotate(env, task, target_quat=target_quat)
        if done:
            print("[Plan] Ended prematurely after rotate")
            return

        # 2) Pull the drawer handle to open it (assumes handle is in reach)
        print("[Plan] Step 2: pull drawer handle to open unlocked drawer")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Plan] Ended prematurely after pull")
            return

        # 3) Pick tomato1 from the table
        print("[Plan] Step 3: pick tomato1")
        obs, reward, done = pick(env, task, target_pos=np.array(tomato1_pos))
        if done:
            print("[Plan] Ended prematurely after pick tomato1")
            return

        # 4) Place tomato1 on the plate
        print("[Plan] Step 4: place tomato1 on plate")
        obs, reward, done = place(env, task, target_pos=np.array(plate_pos))
        if done:
            print("[Plan] Ended prematurely after place tomato1")
            return

        # 5) Pick tomato2 from the table
        print("[Plan] Step 5: pick tomato2")
        obs, reward, done = pick(env, task, target_pos=np.array(tomato2_pos))
        if done:
            print("[Plan] Ended prematurely after pick tomato2")
            return

        # 6) Place tomato2 on the plate
        print("[Plan] Step 6: place tomato2 on plate")
        obs, reward, done = place(env, task, target_pos=np.array(plate_pos))
        if done:
            print("[Plan] Completed all steps; done flag was True")
            return

        print(f"[Plan] Finished all steps; done flag = {done}, reward = {reward}")

    except Exception as e:
        print(f"[Error] Exception during execution: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()