[Frozen Code Start]
    
[Frozen Code End]

import numpy as np
from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Drawer-and-Tomato Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # retrieve all known object poses
        positions = get_object_positions()
        # map PDDL drawer pose names to keys in positions
        side_bottom_pos   = positions['bottom_side_pos']
        anchor_bottom_pos = positions['bottom_anchor_pos']
        tomato1_pos       = positions['tomato1']
        tomato2_pos       = positions['tomato2']
        table_pos         = positions.get('table', None)
        plate_pos         = positions['plate']

        # define quaternions for zero and ninety degrees rotation about z axis
        zero_quat    = np.array([0.0, 0.0, 0.0, 1.0])
        ninety_quat  = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])

        # Step 1: rotate gripper from zero_deg to ninety_deg
        print("[Step 1] rotate gripper to ninety_deg")
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("Task ended prematurely after rotate.")
            return

        # Step 2: move to side position of bottom drawer
        print("[Step 2] move to side position of bottom drawer")
        obs, reward, done = move(env, task, side_bottom_pos)
        if done:
            print("Task ended prematurely after move-to-side.")
            return

        # Step 3: move to anchor position of bottom drawer
        print("[Step 3] move to anchor position of bottom drawer")
        obs, reward, done = move(env, task, anchor_bottom_pos)
        if done:
            print("Task ended prematurely after move-to-anchor.")
            return

        # Step 4: pick the drawer handle (acts as pick-drawer)
        print("[Step 4] pick drawer handle at anchor position")
        obs, reward, done = pick(env, task, anchor_bottom_pos)
        if done:
            print("Task ended prematurely after pick-drawer.")
            return

        # Step 5: pull the drawer open
        print("[Step 5] pull drawer open")
        # pick has closed gripper; pull along x-axis by 0.1m
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("Task ended prematurely after pull.")
            return

        # Step 6: pick tomato1 from table
        print("[Step 6] pick tomato1 from table")
        if table_pos is None:
            raise RuntimeError("table position not found in object positions")
        obs, reward, done = pick(env, task, tomato1_pos)
        if done:
            print("Task ended prematurely after picking tomato1.")
            return

        # Step 7: place tomato1 on plate
        print("[Step 7] place tomato1 on plate")
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("Task ended prematurely after placing tomato1.")
            return

        # Step 8: pick tomato2 from table
        print("[Step 8] pick tomato2 from table")
        obs, reward, done = pick(env, task, tomato2_pos)
        if done:
            print("Task ended prematurely after picking tomato2.")
            return

        # Step 9: place tomato2 on plate
        print("[Step 9] place tomato2 on plate")
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("Task ended prematurely after placing tomato2.")
            return

        print("===== Task Completed: Drawer opened and tomatoes placed on plate =====")

    except Exception as e:
        print(f"Error during task execution: {e}")
    finally:
        shutdown_environment(env)
        print("===== Environment Shutdown =====")

if __name__ == "__main__":
    run_skeleton_task()