import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

# [Frozen Code Start]
# [Frozen Code End]

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # reset environment
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # wrap step and observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # retrieve all object and pose information
        positions = get_object_positions()
        # expected keys: 'bottom_side_pos', 'bottom_anchor_pos', 'tomato1', 'tomato2', 'plate'

        # Step 1: rotate gripper from zero to ninety degrees around its local z‑axis
        print("[Task] Rotating gripper to 90°")
        # build a quaternion that represents 90° around z
        quat90 = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        quat90 = normalize_quaternion(quat90)
        obs, reward, done = rotate(env, task, quat90)
        if done:
            print("[Task] Task ended during rotate")
            return

        # Step 2: move to drawer side position
        side_pos = positions['bottom_side_pos']
        print(f"[Task] Moving to drawer side position at {side_pos}")
        obs, reward, done = move(env, task, target_pos=np.array(side_pos))
        if done:
            print("[Task] Task ended during move-to-side")
            return

        # Step 3: move to drawer anchor position
        anchor_pos = positions['bottom_anchor_pos']
        print(f"[Task] Moving to drawer anchor position at {anchor_pos}")
        obs, reward, done = move(env, task, target_pos=np.array(anchor_pos))
        if done:
            print("[Task] Task ended during move-to-anchor")
            return

        # Step 4: pick the drawer handle (grasp)
        print(f"[Task] Picking drawer handle at {anchor_pos}")
        obs, reward, done = pick(env, task, target_pos=np.array(anchor_pos))
        if done:
            print("[Task] Task ended during pick-drawer")
            return

        # Step 5: pull the drawer open
        print("[Task] Pulling the drawer open")
        # assume pulling along robot x-axis by 0.1 m
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Task] Task ended during pull")
            return

        # Step 6–9: pick each tomato and place on the plate
        plate_pos = positions['plate']
        for name in ['tomato1', 'tomato2']:
            tomato_pos = positions[name]
            print(f"[Task] Picking {name} at {tomato_pos}")
            obs, reward, done = pick(env, task, target_pos=np.array(tomato_pos))
            if done:
                print(f"[Task] Task ended during pick of {name}")
                return

            print(f"[Task] Placing {name} on plate at {plate_pos}")
            obs, reward, done = place(env, task, target_pos=np.array(plate_pos))
            if done:
                print(f"[Task] Task ended during place of {name}")
                return

        print("[Task] Goal achieved: all tomatoes placed on plate")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()