import numpy as np
from env import setup_environment, shutdown_environment
from skill_code import pick, place, move, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        positions = get_object_positions()
        zero_deg_quat   = positions.get('zero_deg')
        ninety_deg_quat = positions.get('ninety_deg')
        side_bottom     = positions.get('bottom_side_pos')
        anchor_bottom   = positions.get('bottom_anchor_pos')
        tomato1_pos     = positions.get('tomato1')
        tomato2_pos     = positions.get('tomato2')
        plate_pos       = positions.get('plate')

        for name, val in [('zero_deg', zero_deg_quat),
                          ('ninety_deg', ninety_deg_quat),
                          ('bottom_side_pos', side_bottom),
                          ('bottom_anchor_pos', anchor_bottom),
                          ('tomato1', tomato1_pos),
                          ('tomato2', tomato2_pos),
                          ('plate', plate_pos)]:
            if val is None:
                raise KeyError(f"Missing position/orientation for '{name}'")

        # Steps 1–6 are frozen below; do not modify
        # [Frozen Code Start]
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
        obs, reward, done = move(env, task, target_pos=np.array(side_bottom))
        obs, reward, done = move(env, task, target_pos=np.array(anchor_bottom))
        obs, reward, done = pick(env, task, target_pos=np.array(anchor_bottom))
        obs, reward, done = pull(env, task, pull_distance=0.2, pull_axis='x')
        obs, reward, done = pick(env, task, target_pos=np.array(tomato1_pos))
        # [Frozen Code End]

        if done:
            print("[Task] Finished early during pick of tomato1; reward:", reward)
            return

        # Step 7: Place tomato1 on plate
        print("[Task] Step 7: Place tomato1 on plate")
        obs, reward, done = place(env, task, target_pos=np.array(plate_pos))
        if done:
            print("[Task] Finished early during place of tomato1; reward:", reward)
            return

        # Step 8: Pick tomato2
        print("[Task] Step 8: Pick tomato2")
        obs, reward, done = pick(env, task, target_pos=np.array(tomato2_pos))
        if done:
            print("[Task] Finished early during pick of tomato2; reward:", reward)
            return

        # Step 9: Place tomato2 on plate
        print("[Task] Step 9: Place tomato2 on plate")
        obs, reward, done = place(env, task, target_pos=np.array(plate_pos))
        if done:
            print("[Task] Finished after placing tomato2; reward:", reward)
        else:
            print("[Task] Completed all steps; final reward:", reward)

    except Exception as e:
        print(f"[Task] Exception encountered: {e}")
    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()