import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        bottom_side_pos = positions['bottom_side_pos']
        bottom_anchor_pos = positions['bottom_anchor_pos']
        tomato1_pos = positions['item1']
        tomato2_pos = positions['item2']
        plate_pos    = positions['plate']

        # === Execute Oracle Plan ===

        # Step 1: rotate gripper from zero_deg to ninety_deg
        half = np.sqrt(2) / 2.0
        target_quat = np.array([half, 0.0, 0.0, half])
        print("[Task] Step 1: rotate gripper to ninety_deg")
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Finished early at step 1 (rotate). Reward:", reward)
            return

        # Step 2: move to the drawer's side position
        print("[Task] Step 2: move to drawer side position")
        obs, reward, done = move(env, task, bottom_side_pos)
        if done:
            print("[Task] Finished early at step 2 (move-to-side). Reward:", reward)
            return

        # Step 3: move to the drawer's anchor position
        print("[Task] Step 3: move to drawer anchor position")
        obs, reward, done = move(env, task, bottom_anchor_pos)
        if done:
            print("[Task] Finished early at step 3 (move-to-anchor). Reward:", reward)
            return

        # Step 4: pick the drawer handle
        print("[Task] Step 4: pick drawer handle")
        obs, reward, done = pick(env, task, bottom_anchor_pos)
        if done:
            print("[Task] Finished early at step 4 (pick-drawer). Reward:", reward)
            return

        # Step 5: pull the drawer open
        print("[Task] Step 5: pull drawer open")
        obs, reward, done = pull(env, task, pull_distance=0.2, pull_axis='x')
        if done:
            print("[Task] Finished early at step 5 (pull). Reward:", reward)
            return

        # Step 6: pick the first tomato
        print("[Task] Step 6: pick first tomato")
        obs, reward, done = pick(env, task, tomato1_pos)
        if done:
            print("[Task] Finished early at step 6 (pick tomato1). Reward:", reward)
            return

        # Step 7: place the first tomato on the plate
        print("[Task] Step 7: place first tomato on plate")
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Task] Finished early at step 7 (place tomato1). Reward:", reward)
            return

        # Step 8: pick the second tomato
        print("[Task] Step 8: pick second tomato")
        obs, reward, done = pick(env, task, tomato2_pos)
        if done:
            print("[Task] Finished early at step 8 (pick tomato2). Reward:", reward)
            return

        # Step 9: place the second tomato on the plate
        print("[Task] Step 9: place second tomato on plate")
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Task] Finished early at step 9 (place tomato2). Reward:", reward)
            return

        print("[Task] Completed all steps successfully. Final reward:", reward)

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()