import numpy as np
from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        bottom_side_pos    = positions['bottom_side_pos']
        bottom_anchor_pos  = positions['bottom_anchor_pos']
        tomato1_pos        = positions['tomato1']
        tomato2_pos        = positions['tomato2']
        plate_pos          = positions['plate']
        ninety_deg_quat    = positions.get('ninety_deg', None)
        if ninety_deg_quat is None:
            ninety_deg_quat = obs.gripper_pose[3:7]

        # === Execute Oracle Plan Steps ===
        # Frozen Code Start
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos)
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        # Frozen Code End

        # Step 6: pick tomato1
        print("[Task] Step 6: pick tomato1")
        obs, reward, done = pick(env, task, target_pos=tomato1_pos)
        if done:
            print("[Task] Early exit after pick tomato1. Reward:", reward)
            return

        # Step 7: place tomato1 on plate
        print("[Task] Step 7: place tomato1 on plate")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Early exit after place tomato1. Reward:", reward)
            return

        # Step 8: pick tomato2
        print("[Task] Step 8: pick tomato2")
        obs, reward, done = pick(env, task, target_pos=tomato2_pos)
        if done:
            print("[Task] Early exit after pick tomato2. Reward:", reward)
            return

        # Step 9: place tomato2 on plate
        print("[Task] Step 9: place tomato2 on plate")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Completed all steps successfully. Reward:", reward)
        else:
            print("[Task] Plan executed but done=False. Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()