from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions
import numpy as np
from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video recording if desired
        init_video_writers(obs)

        # Wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve object positions
        positions = get_object_positions()

        # === Execute Oracle Plan ===

        # Step 1: Rotate gripper to ninety degrees around Z axis
        print("[Plan] Step 1: Rotate gripper to ninety_deg")
        ninety_quat = R.from_euler('z', 90, degrees=True).as_quat()
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("[Task] Ended after rotate.")
            return

        # Step 2: Move to bottom side position
        print("[Plan] Step 2: Move to bottom side position")
        side_bottom_pos = np.array(positions['bottom_side_pos'])
        obs, reward, done = move(env, task, side_bottom_pos)
        if done:
            print("[Task] Ended after move to side-pos-bottom.")
            return

        # Step 3: Move to bottom anchor position
        print("[Plan] Step 3: Move to bottom anchor position")
        anchor_bottom_pos = np.array(positions['bottom_anchor_pos'])
        obs, reward, done = move(env, task, anchor_bottom_pos)
        if done:
            print("[Task] Ended after move to anchor-pos-bottom.")
            return

        # Step 4: Pull drawer open
        print("[Plan] Step 4: Pull drawer open")
        # We assume a pull distance of 0.1 along the X axis to open drawer
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Task] Ended after pull.")
            return

        # Step 5: Pick tomato1 (item1) from table
        print("[Plan] Step 5: Pick tomato1 from table")
        pos_item1 = np.array(positions['item1'])
        obs, reward, done = pick(env, task, pos_item1)
        if done:
            print("[Task] Ended after picking tomato1.")
            return

        # Step 6: Place tomato1 on plate
        print("[Plan] Step 6: Place tomato1 on plate")
        plate_pos = np.array(positions['plate'])
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Task] Ended after placing tomato1.")
            return

        # Step 7: Pick tomato2 (item2) from table
        print("[Plan] Step 7: Pick tomato2 from table")
        pos_item2 = np.array(positions['item2'])
        obs, reward, done = pick(env, task, pos_item2)
        if done:
            print("[Task] Ended after picking tomato2.")
            return

        # Step 8: Place tomato2 on plate
        print("[Plan] Step 8: Place tomato2 on plate")
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Task] Ended after placing tomato2.")
            return

        print("[Task] Plan executed successfully. Final reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()