
import numpy as np
from scipy.spatial.transform import Rotation as R
from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers and wrap step/get_observation for recording
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve object positions from the environment
        positions = get_object_positions()
        bottom_side_pos   = positions['bottom_side_pos']
        bottom_anchor_pos = positions['bottom_anchor_pos']
        tomato1_pos       = positions['tomato1']
        tomato2_pos       = positions['tomato2']
        plate_pos         = positions['plate']

        # Step 1: rotate gripper from zero_deg to ninety_deg
        print("[Plan] Step 1: rotate to ninety_deg")
        obs = task.get_observation()
        init_quat = normalize_quaternion(obs.gripper_pose[3:7])
        rot_z90 = R.from_euler('z', 90, degrees=True)
        target_quat = (rot_z90 * R.from_quat(init_quat)).as_quat()
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Plan] Early termination after step 1.")
            return

        # Step 2: move to drawer side position
        print("[Plan] Step 2: move to drawer side position")
        obs, reward, done = move(env, task, target_pos=np.array(bottom_side_pos))
        if done:
            print("[Plan] Early termination after step 2.")
            return

        # Step 3: move to drawer anchor position
        print("[Plan] Step 3: move to drawer anchor position")
        obs, reward, done = move(env, task, target_pos=np.array(bottom_anchor_pos))
        if done:
            print("[Plan] Early termination after step 3.")
            return

        # Step 4: pick drawer handle
        print("[Plan] Step 4: pick drawer handle")
        obs, reward, done = pick(env, task, target_pos=np.array(bottom_anchor_pos))
        if done:
            print("[Plan] Early termination after step 4.")
            return

        # Step 5: pull to open drawer
        print("[Plan] Step 5: pull to open drawer")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Plan] Early termination after step 5.")
            return

        # Step 6: pick the first tomato
        print("[Plan] Step 6: pick tomato1")
        obs, reward, done = pick(env, task, target_pos=np.array(tomato1_pos))
        if done:
            print("[Plan] Early termination after step 6.")
            return

        # Step 7: place tomato1 on the plate
        print("[Plan] Step 7: place tomato1 on plate")
        obs, reward, done = place(env, task, target_pos=np.array(plate_pos))
        if done:
            print("[Plan] Early termination after step 7.")
            return

        # Step 8: pick the second tomato
        print("[Plan] Step 8: pick tomato2")
        obs, reward, done = pick(env, task, target_pos=np.array(tomato2_pos))
        if done:
            print("[Plan] Early termination after step 8.")
            return

        # Step 9: place tomato2 on the plate
        print("[Plan] Step 9: place tomato2 on plate")
        obs, reward, done = place(env, task, target_pos=np.array(plate_pos))
        if done:
            print("[Plan] Early termination after step 9.")
            return

        print("[Plan] All steps executed. Drawer opened and both tomatoes placed on the plate.")

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
