import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from scipy.spatial.transform import Rotation as R

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for recording
        init_video_writers(obs)

        # Wrap the task step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        tomato1_pos = positions['tomato1']
        tomato2_pos = positions['tomato2']
        plate_pos = positions['plate']
        bottom_side_pos = positions['bottom_side_pos']
        bottom_anchor_pos = positions['bottom_anchor_pos']

        # === Precompute Rotation Quaternion for 90° about Y axis ===
        ninety_deg_quat = R.from_euler('y', 90, degrees=True).as_quat()

        # === Execute Oracle Plan ===

        # Frozen Code Start
        obs, reward, done = rotate(env, task, ninety_deg_quat)
        obs, reward, done = move(env, task, bottom_side_pos)
        # Frozen Code End

        if done:
            print("[Task] Ended after move-to-side")
            return

        # Step 3: Move to the anchor position of the bottom drawer
        obs, reward, done = move(env, task, bottom_anchor_pos)
        if done:
            print("[Task] Ended after move-to-anchor")
            return

        # Step 4: Pick the drawer handle at the anchor position
        obs, reward, done = pick(env, task, bottom_anchor_pos)
        if done:
            print("[Task] Ended after pick-drawer")
            return

        # Step 5: Pull drawer open by 0.15m along the x axis
        obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='x')
        if done:
            print("[Task] Ended after pull")
            return

        # Step 6: Pick tomato1 from table
        obs, reward, done = pick(env, task, tomato1_pos)
        if done:
            print("[Task] Ended after pick tomato1")
            return

        # Step 7: Place tomato1 on plate
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Task] Ended after place tomato1")
            return

        # Step 8: Pick tomato2 from table
        obs, reward, done = pick(env, task, tomato2_pos)
        if done:
            print("[Task] Ended after pick tomato2")
            return

        # Step 9: Place tomato2 on plate
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Task] Ended after place tomato2")
            return

        print("[Task] Completed successfully! Final reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()