import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # wrap for video recording / logging
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        print("[Task] Executing oracle plan: Open bottom drawer and place all tomatoes on the plate.")

        # get positions
        positions = get_object_positions()
        try:
            side_pos       = positions['bottom_side_pos']
            anchor_pos     = positions['bottom_anchor_pos']
            tomato1_pos    = positions['tomato1']
            tomato2_pos    = positions['tomato2']
            plate_pos      = positions['plate']
        except KeyError as e:
            print(f"[Error] Missing position for {e}. Cannot execute plan.")
            return

        # define quaternions
        zero_quat   = np.array([0.0, 0.0, 0.0, 1.0])
        ninety_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])

        # Step 1: rotate gripper from zero to ninety
        print("[Step 1] rotate to ninety_deg")
        obs, reward, done = rotate(env, task, target_quat=ninety_quat)
        if done:
            print("[Task] Terminated during rotation.")
            return

        # Step 2: move to the side position of bottom drawer
        print("[Step 2] move to bottom_side_pos")
        obs, reward, done = move(env, task, target_pos=side_pos)
        if done:
            print("[Task] Terminated during move-to-side.")
            return

        # Step 3: move to the anchor position of bottom drawer
        print("[Step 3] move to bottom_anchor_pos")
        obs, reward, done = move(env, task, target_pos=anchor_pos)
        if done:
            print("[Task] Terminated during move-to-anchor.")
            return

        # Step 4: pick the drawer handle
        print("[Step 4] pick drawer handle")
        obs, reward, done = pick(env, task, target_pos=anchor_pos)
        if done:
            print("[Task] Terminated during pick-drawer.")
            return

        # Step 5: pull the drawer open
        print("[Step 5] pull drawer open")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Task] Terminated during pull.")
            return

        # Step 6: pick tomato1 from the table
        print("[Step 6] pick tomato1")
        obs, reward, done = pick(env, task, target_pos=tomato1_pos)
        if done:
            print("[Task] Terminated during pick tomato1.")
            return

        # Step 7: place tomato1 on plate
        print("[Step 7] place tomato1 on plate")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Terminated during place tomato1.")
            return

        # Step 8: pick tomato2 from the table
        print("[Step 8] pick tomato2")
        obs, reward, done = pick(env, task, target_pos=tomato2_pos)
        if done:
            print("[Task] Terminated during pick tomato2.")
            return

        # Step 9: place tomato2 on plate
        print("[Step 9] place tomato2 on plate")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Terminated during place tomato2.")
            return

        print(f"[Task] Oracle plan completed. Final reward: {reward}, done={done}")

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()