import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Expected keys: 'bottom_side_pos', 'bottom_anchor_pos', 'tomato1', 'tomato2', 'plate'

        # === Execute Oracle Plan ===

        # Step 1: rotate gripper from zero_deg to ninety_deg about Z
        print("[Plan Step 1] rotate to ninety_deg")
        # Convert 90° around Z to quaternion (xyzw)
        target_quat = R.from_euler('z', 90, degrees=True).as_quat()
        obs, reward, done = rotate(env, task, target_quat=target_quat)
        if done:
            print("[Plan] Terminated after rotate.")
            return

        # Step 2: move-to-side: move gripper to the side position of bottom drawer
        print("[Plan Step 2] move to side of bottom drawer")
        side_pos = np.array(positions['bottom_side_pos'])
        obs, reward, done = move(env, task, target_pos=side_pos)
        if done:
            print("[Plan] Terminated after move-to-side.")
            return

        # Step 3: move-to-anchor: move gripper to the anchor position of bottom drawer
        print("[Plan Step 3] move to anchor of bottom drawer")
        anchor_pos = np.array(positions['bottom_anchor_pos'])
        obs, reward, done = move(env, task, target_pos=anchor_pos)
        if done:
            print("[Plan] Terminated after move-to-anchor.")
            return

        # Step 4: pick-drawer (grasp drawer handle at anchor position)
        print("[Plan Step 4] pick drawer handle")
        obs, reward, done = pick(env, task, target_pos=anchor_pos)
        if done:
            print("[Plan] Terminated after pick-drawer.")
            return

        # Step 5: pull drawer open
        print("[Plan Step 5] pull drawer open")
        # Pull distance chosen to open the drawer sufficiently
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Plan] Terminated after pull.")
            return

        # Step 6: pick tomato1 from table
        print("[Plan Step 6] pick tomato1 from table")
        tomato1_pos = np.array(positions['tomato1'])
        obs, reward, done = pick(env, task, target_pos=tomato1_pos)
        if done:
            print("[Plan] Terminated after picking tomato1.")
            return

        # Step 7: place tomato1 on plate
        print("[Plan Step 7] place tomato1 on plate")
        plate_pos = np.array(positions['plate'])
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Plan] Terminated after placing tomato1.")
            return

        # Step 8: pick tomato2 from table
        print("[Plan Step 8] pick tomato2 from table")
        tomato2_pos = np.array(positions['tomato2'])
        obs, reward, done = pick(env, task, target_pos=tomato2_pos)
        if done:
            print("[Plan] Terminated after picking tomato2.")
            return

        # Step 9: place tomato2 on plate
        print("[Plan Step 9] place tomato2 on plate")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Plan] Terminated after placing tomato2.")
            return

        # Final check
        print("[Plan] All steps executed. Task completed with reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()