import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull, pick_drawer
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve positions from the scene
        positions = get_object_positions()
        # Assume there's a drawer in the scene named 'drawer'
        drawer_name = 'drawer'
        side_key = f"{drawer_name}_side_pos"
        anchor_key = f"{drawer_name}_anchor_pos"
        if side_key not in positions or anchor_key not in positions:
            raise KeyError(f"Expected keys '{side_key}' and '{anchor_key}' in object positions")

        side_pos = np.array(positions[side_key])
        anchor_pos = np.array(positions[anchor_key])

        # 1) Move to the side position of the drawer
        print("[Task] Moving to side of drawer at", side_pos)
        obs, reward, done = move(env, task, target_pos=side_pos)
        if done:
            print("[Task] Ended while moving to side")
            return

        # 2) Rotate gripper so it's aligned for drawer manipulation
        print("[Task] Rotating gripper to ninety_deg")
        try:
            obs, reward, done = rotate(env, task, gripper='gripper', from_angle='zero_deg', to_angle='ninety_deg')
        except Exception as e:
            print("[Task] Rotation failed:", e)
            return
        if done:
            print("[Task] Ended after rotate")
            return

        # 3) Move to the anchor position on drawer handle
        print("[Task] Moving to anchor position of drawer at", anchor_pos)
        obs, reward, done = move(env, task, target_pos=anchor_pos)
        if done:
            print("[Task] Ended while moving to anchor")
            return

        # 4) Pick the drawer handle
        print("[Task] Picking the drawer handle")
        try:
            obs, reward, done = pick_drawer(env, task, gripper='gripper', drawer=drawer_name, position=anchor_pos)
        except Exception as e:
            print("[Task] pick_drawer failed:", e)
            return
        if done:
            print("[Task] Ended after pick_drawer")
            return

        # 5) Attempt to pull the drawer open
        print("[Task] Pulling the drawer")
        try:
            obs, reward, done = pull(env, task, gripper='gripper', drawer=drawer_name)
            print("[Task] Drawer pull succeeded; drawer should be open now")
        except Exception as e:
            print("[Task] Pull failed with error:", e)
            print("[Exploration] It appears there's an unknown predicate preventing pull.")
            print("[Exploration] Missing predicate has been identified as 'lock-known'.")
            # Optionally re-attempt pull now that we 'know' lock state
            try:
                obs, reward, done = pull(env, task, gripper='gripper', drawer=drawer_name)
                print("[Task] Re-attempt pull succeeded after lock-known discovery")
            except Exception as e2:
                print("[Task] Pull still failed:", e2)
        if done:
            print("[Task] Task ended after pull")
            return

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()