import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Get the positions of relevant objects in the scene
        positions = get_object_positions()
        # Expect keys: 'drawer_side_pos', 'drawer_anchor_pos', 'tomato_pos', 'bowl_pos'
        side_pos = np.array(positions['drawer_side_pos'])
        anchor_pos = np.array(positions['drawer_anchor_pos'])
        tomato_pos = np.array(positions['tomato_pos'])
        bowl_pos = np.array(positions['bowl_pos'])

        # The name of the gripper we pass into the primitive calls
        gripper_name = 'gripper'  # adjust if your setup uses a different identifier
        drawer_name = 'drawer'    # adjust to your object name

        # 1) Move gripper near the side of the drawer
        print("[Task] Moving to side of drawer at", side_pos)
        obs, reward, done = move(env, task, target_pos=side_pos)
        if done:
            print("[Task] Task ended prematurely after moving to drawer side.")
            return

        # 2) Rotate the gripper to ninety degrees to align with handle
        print("[Task] Rotating gripper to ninety_deg")
        obs, reward, done = rotate(env, task, gripper_name, 'zero_deg', 'ninety_deg')
        if done:
            print("[Task] Task ended prematurely during rotate.")
            return

        # 3) Approach handle side position on the drawer
        print("[Task] Sliding gripper along side to handle")
        obs, reward, done = move_to_side(env, task, gripper_name, drawer_name, side_pos)
        if done:
            print("[Task] Task ended prematurely during move_to_side.")
            return

        # 4) Move gripper from side to anchor position on the handle
        print("[Task] Moving gripper to anchor position on handle", anchor_pos)
        obs, reward, done = move_to_anchor(env, task, gripper_name, drawer_name, anchor_pos)
        if done:
            print("[Task] Task ended prematurely during move_to_anchor.")
            return

        # 5) Pick the drawer handle
        print("[Task] Grasping drawer handle")
        obs, reward, done = pick_drawer(env, task, gripper_name, drawer_name, anchor_pos)
        if done:
            print("[Task] Task ended prematurely during pick_drawer.")
            return

        # 6) Pull the drawer open
        print("[Task] Pulling drawer open")
        obs, reward, done = pull(env, task, gripper_name, drawer_name)
        if done:
            print("[Task] Task ended prematurely during pull.")
            return

        # 7) Now inside the open drawer, pick the tomato
        print("[Task] Picking tomato at", tomato_pos)
        obs, reward, done = pick(env, task, target_pos=tomato_pos, approach_distance=0.15,
                                   max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        if done:
            print("[Task] Task ended prematurely during pick(tomato).")
            return

        # 8) Place the tomato into the bowl
        print("[Task] Placing tomato into bowl at", bowl_pos)
        obs, reward, done = place(env, task, target_pos=bowl_pos, approach_distance=0.15,
                                    max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
        if done:
            print("[Task] Task ended prematurely during place(tomato).")
            return

        print("[Task] Completed all steps successfully.")
    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()