import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers if desired
        init_video_writers(obs)

        # Wrap step and observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        print("[Info] Retrieved object positions:", positions)

        # === Exploration Phase: Find Missing Predicate ===
        # We will try pick and pull actions on each object and catch errors that indicate
        # which predicate is missing (weight-known, durability-known, lock-known).
        missing_predicate = None

        for name, pos in positions.items():
            if missing_predicate is not None:
                break

            print(f"[Explore] Testing object '{name}' at position {pos}")

            # First try to pick the object to see if weight-known or durability-known is missing
            if name.startswith("drawer_handle") or name.startswith("drawer"):
                # test pull on drawer handle
                try:
                    # Approach the handle
                    obs, reward, done = move(
                        env,
                        task,
                        target_pos=pos,
                        approach_distance=0.10,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=5.0
                    )
                    if done:
                        print("[Explore] Episode ended prematurely during move to drawer handle.")
                        break

                    # Pick the drawer handle
                    obs, reward, done = pick(
                        env,
                        task,
                        target_pos=pos,
                        approach_distance=0.02,
                        max_steps=80,
                        threshold=0.005,
                        approach_axis='z',
                        timeout=5.0
                    )
                    # Now attempt pull: this will reveal lock-known predicate
                    obs, reward, done = pull(env, task)
                    print("[Explore] Pull succeeded without exception.")
                except Exception as e:
                    msg = str(e)
                    print(f"[Explore] Caught exception during pull: {msg}")
                    if "lock-known" in msg:
                        missing_predicate = "lock-known"
                    else:
                        missing_predicate = "unknown_lock_predicate"
            else:
                # test pick on a generic object
                try:
                    obs, reward, done = move(
                        env,
                        task,
                        target_pos=pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=5.0
                    )
                    if done:
                        print("[Explore] Episode ended prematurely during move to object.")
                        break

                    obs, reward, done = pick(
                        env,
                        task,
                        target_pos=pos,
                        approach_distance=0.05,
                        max_steps=80,
                        threshold=0.005,
                        approach_axis='z',
                        timeout=5.0
                    )
                    print(f"[Explore] Pick succeeded on '{name}' without exception.")
                except Exception as e:
                    msg = str(e)
                    print(f"[Explore] Caught exception during pick: {msg}")
                    if "weight-known" in msg:
                        missing_predicate = "weight-known"
                    elif "durability-known" in msg:
                        missing_predicate = "durability-known"
                    else:
                        missing_predicate = "unknown_pick_predicate"
        
        if missing_predicate:
            print(f"[Result] Identified missing predicate: {missing_predicate}")
        else:
            print("[Result] No missing predicate detected during exploration.")
        
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()