import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers (optional)
        init_video_writers(obs)

        # Wrap step and get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Example expected keys in positions:
        #   'object_1', 'object_2', 'bin', 'drawer_handle', 'drawer_target_quat'
        # Adjust these to match your environment
        object_1_pos = positions.get('object_1')
        object_2_pos = positions.get('object_2')
        bin_pos      = positions.get('bin')
        drawer_handle_pos = positions.get('drawer_handle')
        drawer_target_quat = positions.get('drawer_target_quat')

        # === Exploration Phase ===
        # Try to approach each object to identify which predicate is missing
        print("[Exploration] Starting to move to known object locations to identify environment predicates.")
        for name, pos in positions.items():
            if pos is None:
                continue
            try:
                print(f"[Exploration] Moving to {name} at {pos}")
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=pos,
                    max_steps=100,
                    threshold=0.01,
                    timeout=5.0
                )
                if done:
                    print(f"[Exploration] Task ended unexpectedly while moving to {name}.")
                    return
            except Exception as e:
                print(f"[Exploration] Error when moving to {name}: {e}")
        
        # Attempt to pull the drawer handle to see if lock-known predicate is needed
        print("[Exploration] Attempting pull on drawer handle to detect missing predicate.")
        try:
            obs, reward, done = pull(
                env,
                task,
                target_pos=drawer_handle_pos,
                max_steps=100,
                threshold=0.01,
                timeout=5.0
            )
            print("[Exploration] Pull succeeded; lock-known predicate appears supported.")
        except Exception as e:
            print(f"[Exploration] Pull failed with error: {e}")
            print("[Exploration] Likely missing predicate: lock-known")
        
        # === Main Task Plan ===
        # 1) Pick object_1 and place into the bin
        print("[Task] Picking up object_1 at", object_1_pos)
        obs, reward, done = pick(
            env,
            task,
            target_pos=object_1_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick(object_1).")
            return

        print("[Task] Placing object_1 into bin at", bin_pos)
        obs, reward, done = place(
            env,
            task,
            target_pos=bin_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after place(object_1).")
            return

        # 2) Pick object_2 and place into the bin
        print("[Task] Picking up object_2 at", object_2_pos)
        obs, reward, done = pick(
            env,
            task,
            target_pos=object_2_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick(object_2).")
            return

        print("[Task] Placing object_2 into bin at", bin_pos)
        obs, reward, done = place(
            env,
            task,
            target_pos=bin_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after place(object_2).")
            return

        print("===== Main Task Completed Successfully =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()