import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with an exploration phase to identify missing predicates.'''
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset and wrap for recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve positions of all objects in the scene
        positions = get_object_positions()
        print("[Exploration] Retrieved object positions:", positions)

        # Exploration phase: try each skill on each object to detect missing predicates
        missing_predicates = set()
        for obj_name, pos in positions.items():
            print(f"[Exploration] Testing on object '{obj_name}' at position {pos}")

            # Test move
            try:
                print(f"[Exploration] move -> {obj_name}")
                obs, reward, done = move(env, task, target_pos=pos, max_steps=50, threshold=0.01, approach_axis='z', timeout=5.0)
            except Exception as e:
                err = str(e)
                print(f"[Exploration] move error on {obj_name}: {err}")
                # look for any predicate-like keywords
                for kw in ['identified', 'temperature-known', 'durability-known', 'weight-known', 'lock-known']:
                    if kw in err:
                        missing_predicates.add(kw)

            # Test pick (if handempty)
            try:
                print(f"[Exploration] pick -> {obj_name}")
                obs, reward, done = pick(env, task,
                                         target_pos=pos,
                                         approach_distance=0.10,
                                         max_steps=50,
                                         threshold=0.01,
                                         approach_axis='z',
                                         timeout=5.0)
            except Exception as e:
                err = str(e)
                print(f"[Exploration] pick error on {obj_name}: {err}")
                for kw in ['identified', 'temperature-known', 'durability-known', 'weight-known', 'lock-known']:
                    if kw in err:
                        missing_predicates.add(kw)

            # Test pull (only meaningful for potential drawers or handles)
            try:
                print(f"[Exploration] pull -> {obj_name}")
                obs, reward, done = pull(env, task)
            except Exception as e:
                err = str(e)
                print(f"[Exploration] pull error on {obj_name}: {err}")
                if 'lock-known' in err:
                    missing_predicates.add('lock-known')

        # Report any missing predicates found during exploration
        if missing_predicates:
            print("[Exploration] Missing predicates identified:", missing_predicates)
        else:
            print("[Exploration] No missing predicates detected during exploration.")

        # (Optional) Here you could proceed to adjust your domain or retry planning
        # For now we end after exploration
        print("===== Exploration Phase Completed =====")

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()